<?php
defined('BASEPATH') or exit('No direct script access allowed');

class M_hugoffers extends CI_Model
{
    /**
     * Get all enabled HugOffers offers
     * @return array
     */
    public function getEnabledOffers()
    {
        $this->db->where('is_enabled', 1);
        $this->db->where('status', 'active');
        $this->db->order_by('offer_name', 'ASC');
        return $this->db->get('hugoffers_offers')->result_array();
    }

    /**
     * Get all HugOffers offers for admin
     * @param int $limit
     * @param int $offset
     * @return array
     */
    public function getAllOffers($limit = 50, $offset = 0)
    {
        $this->db->order_by('created_at', 'DESC');
        if ($limit > 0) {
            $this->db->limit($limit, $offset);
        }
        return $this->db->get('hugoffers_offers')->result_array();
    }

    /**
     * Get offer by campaign ID
     * @param string $campid
     * @return array|null
     */
    public function getOfferByCampId($campid)
    {
        $this->db->where('campid', $campid);
        $result = $this->db->get('hugoffers_offers');
        return $result->num_rows() > 0 ? $result->row_array() : null;
    }

    /**
     * Get offer by ID
     * @param int $id
     * @return array|null
     */
    public function getOfferById($id)
    {
        $this->db->where('id', $id);
        $result = $this->db->get('hugoffers_offers');
        return $result->num_rows() > 0 ? $result->row_array() : null;
    }

    /**
     * Add new offer
     * @param array $data
     * @return int|false
     */
    public function addOffer($data)
    {
        // Check if campaign ID already exists
        if ($this->getOfferByCampId($data['campid'])) {
            return false;
        }

        $insert_data = array(
            'campid' => $data['campid'],
            'offer_name' => $data['offer_name'],
            'price' => floatval($data['price']),
            'daily_cap' => intval($data['daily_cap']),
            'app_desc' => isset($data['app_desc']) ? $data['app_desc'] : '',
            'preview_link' => isset($data['preview_link']) ? $data['preview_link'] : '',
            'custom_payout' => isset($data['custom_payout']) ? floatval($data['custom_payout']) : null,
            'platform' => isset($data['platform']) ? $data['platform'] : 'All',
            'geo' => isset($data['geo']) ? $data['geo'] : 'ALL',
            'app_category' => isset($data['app_category']) ? $data['app_category'] : 'General',
            'requirement_traffic' => isset($data['requirement_traffic']) ? $data['requirement_traffic'] : '',
            'icon_link' => isset($data['icon_link']) ? $data['icon_link'] : '',
            'app_name' => isset($data['app_name']) ? $data['app_name'] : '',
            'app_size' => isset($data['app_size']) ? $data['app_size'] : '',
            'app_rate' => isset($data['app_rate']) ? floatval($data['app_rate']) : 0.0,
            'app_id' => isset($data['app_id']) ? $data['app_id'] : '',
            'images' => isset($data['images']) ? json_encode($data['images']) : '',
            'videos' => isset($data['videos']) ? json_encode($data['videos']) : '',
            'added_by' => isset($data['added_by']) ? intval($data['added_by']) : null,
            'is_enabled' => 1,
            'status' => 'active'
        );

        if ($this->db->insert('hugoffers_offers', $insert_data)) {
            return $this->db->insert_id();
        }
        
        return false;
    }

    /**
     * Update offer
     * @param int $id
     * @param array $data
     * @return bool
     */
    public function updateOffer($id, $data)
    {
        $update_data = array();
        
        if (isset($data['offer_name'])) $update_data['offer_name'] = $data['offer_name'];
        if (isset($data['price'])) $update_data['price'] = floatval($data['price']);
        if (isset($data['daily_cap'])) $update_data['daily_cap'] = intval($data['daily_cap']);
        if (isset($data['app_desc'])) $update_data['app_desc'] = $data['app_desc'];
        if (isset($data['preview_link'])) $update_data['preview_link'] = $data['preview_link'];
        if (isset($data['custom_payout'])) $update_data['custom_payout'] = $data['custom_payout'] ? floatval($data['custom_payout']) : null;
        if (isset($data['status'])) $update_data['status'] = $data['status'];
        if (isset($data['is_enabled'])) $update_data['is_enabled'] = intval($data['is_enabled']);
        if (isset($data['platform'])) $update_data['platform'] = $data['platform'];
        if (isset($data['geo'])) $update_data['geo'] = $data['geo'];
        if (isset($data['app_category'])) $update_data['app_category'] = $data['app_category'];
        if (isset($data['requirement_traffic'])) $update_data['requirement_traffic'] = $data['requirement_traffic'];
        if (isset($data['icon_link'])) $update_data['icon_link'] = $data['icon_link'];
        if (isset($data['app_name'])) $update_data['app_name'] = $data['app_name'];
        if (isset($data['app_size'])) $update_data['app_size'] = $data['app_size'];
        if (isset($data['app_rate'])) $update_data['app_rate'] = floatval($data['app_rate']);
        if (isset($data['app_id'])) $update_data['app_id'] = $data['app_id'];
        if (isset($data['images'])) $update_data['images'] = is_array($data['images']) ? json_encode($data['images']) : $data['images'];
        if (isset($data['videos'])) $update_data['videos'] = is_array($data['videos']) ? json_encode($data['videos']) : $data['videos'];

        if (empty($update_data)) {
            return false;
        }

        $this->db->where('id', $id);
        return $this->db->update('hugoffers_offers', $update_data);
    }

    /**
     * Delete offer
     * @param int $id
     * @return bool
     */
    public function deleteOffer($id)
    {
        $this->db->where('id', $id);
        return $this->db->delete('hugoffers_offers');
    }

    /**
     * Sync offer from API
     * @param string $campid
     * @param string $api_key
     * @return bool
     */
    public function syncOfferFromAPI($campid, $api_key)
    {
        // Load CodeIgniter instance to use helpers
        $CI =& get_instance();
        $CI->load->helper('hugoffers_helper');
        
        $api_offer = get_hugoffers_offer_by_id($api_key, $campid);
        
        if (!$api_offer) {
            return false;
        }

        $existing_offer = $this->getOfferByCampId($campid);
        
        $sync_data = array(
            'offer_name' => $api_offer['offer_name'],
            'platform' => $api_offer['platform'],
            'tracking_link' => $api_offer['tracking_link'],
            'geo' => $api_offer['geo'],
            'preview_link' => $api_offer['preview_link'],
            'price' => $api_offer['price'],
            'price_model' => $api_offer['price_model'],
            'app_category' => $api_offer['app_category'],
            'daily_cap' => $api_offer['daily_cap'],
            'icon_link' => $api_offer['icon_link'],
            'images' => $api_offer['images'],
            'videos' => $api_offer['videos'],
            'app_name' => $api_offer['app_name'],
            'app_desc' => $api_offer['app_desc'],
            'app_size' => $api_offer['app_size'],
            'app_rate' => $api_offer['app_rate'],
            'app_id' => $api_offer['app_id'],
            'requirement_traffic' => $api_offer['requirement_traffic'],
            'status' => $api_offer['status']
        );

        if ($existing_offer) {
            return $this->updateOffer($existing_offer['id'], $sync_data);
        } else {
            $sync_data['campid'] = $campid;
            return $this->addOffer($sync_data);
        }
    }

    /**
     * Increment conversion count
     * @param string $campid
     * @return bool
     */
    public function incrementConversion($campid)
    {
        $this->db->where('campid', $campid);
        $this->db->set('conversion_count', 'conversion_count + 1', FALSE);
        return $this->db->update('hugoffers_offers');
    }

    /**
     * Get offers count
     * @return int
     */
    public function getOffersCount()
    {
        return $this->db->count_all('hugoffers_offers');
    }

    /**
     * Get offers statistics
     * @return array
     */
    public function getOffersStats()
    {
        $stats = array(
            'total_offers' => $this->db->count_all('hugoffers_offers'),
            'active_offers' => 0,
            'inactive_offers' => 0,
            'total_conversions' => 0
        );

        // Count active/inactive offers
        $this->db->select('status, COUNT(*) as count');
        $this->db->group_by('status');
        $status_counts = $this->db->get('hugoffers_offers')->result_array();
        
        foreach ($status_counts as $status) {
            if ($status['status'] == 'active') {
                $stats['active_offers'] = intval($status['count']);
            } else {
                $stats['inactive_offers'] += intval($status['count']);
            }
        }

        // Get total conversions
        $this->db->select_sum('conversion_count');
        $conversion_result = $this->db->get('hugoffers_offers')->row_array();
        $stats['total_conversions'] = intval($conversion_result['conversion_count']);

        return $stats;
    }
} 