<?php
defined('BASEPATH') or exit('No direct script access allowed');

class Hugoffers extends Admin_Controller
{
    public function __construct()
    {
        parent::__construct();
        $this->load->model('M_hugoffers');
        $this->load->helper('hugoffers_helper');
        $this->load->library('form_validation');
    }

    /**
     * List all HugOffers offers
     */
    public function index()
    {
        $this->data['page'] = 'HugOffers Management';
        
        // Get pagination
        $page = isset($_GET['page']) ? intval($_GET['page']) : 1;
        $per_page = 20;
        $offset = ($page - 1) * $per_page;
        
        $this->data['offers'] = $this->M_hugoffers->getAllOffers($per_page, $offset);
        $this->data['total_offers'] = $this->M_hugoffers->getOffersCount();
        $this->data['stats'] = $this->M_hugoffers->getOffersStats();
        
        // Pagination
        $this->data['pagination'] = array(
            'current_page' => $page,
            'per_page' => $per_page,
            'total_pages' => ceil($this->data['total_offers'] / $per_page),
            'total_records' => $this->data['total_offers']
        );
        
        $this->render('hugoffers_manage', $this->data);
    }

    /**
     * Add new offer form
     */
    public function add()
    {
        $this->data['page'] = 'Add HugOffers Offer';
        
        if ($this->input->post()) {
            $this->form_validation->set_rules('campid', 'Campaign ID', 'required|trim');
            $this->form_validation->set_rules('offer_name', 'Offer Name', 'required|trim');
            $this->form_validation->set_rules('price', 'Price (USD)', 'required|numeric');
            $this->form_validation->set_rules('daily_cap', 'Daily Cap', 'required|integer');
            
            if ($this->form_validation->run()) {
                $offer_data = array(
                    'campid' => $this->input->post('campid'),
                    'offer_name' => $this->input->post('offer_name'),
                    'price' => $this->input->post('price'),
                    'daily_cap' => $this->input->post('daily_cap'),
                    'app_desc' => $this->input->post('app_desc'),
                    'preview_link' => $this->input->post('preview_link'),
                    'custom_payout' => $this->input->post('custom_payout'),
                    'platform' => $this->input->post('platform'),
                    'geo' => $this->input->post('geo'),
                    'app_category' => $this->input->post('app_category'),
                    'requirement_traffic' => $this->input->post('requirement_traffic'),
                    'icon_link' => $this->input->post('icon_link'),
                    'app_name' => $this->input->post('app_name'),
                    'app_size' => $this->input->post('app_size'),
                    'app_rate' => $this->input->post('app_rate'),
                    'app_id' => $this->input->post('app_id'),
                    'added_by' => $this->data['admin']['id']
                );
                
                $offer_id = $this->M_hugoffers->addOffer($offer_data);
                
                if ($offer_id) {
                    $this->session->set_flashdata('message', 
                        '<div class="alert alert-success">Offer added successfully!</div>');
                    redirect('admin/hugoffers');
                } else {
                    $this->session->set_flashdata('message', 
                        '<div class="alert alert-danger">Failed to add offer. Campaign ID may already exist.</div>');
                }
            }
        }
        
        $this->render('hugoffers_add', $this->data);
    }

    /**
     * Edit offer
     */
    public function edit($id = 0)
    {
        $offer = $this->M_hugoffers->getOfferById($id);
        if (!$offer) {
            show_404();
        }
        
        $this->data['page'] = 'Edit HugOffers Offer';
        $this->data['offer'] = $offer;
        
        if ($this->input->post()) {
            $this->form_validation->set_rules('offer_name', 'Offer Name', 'required|trim');
            $this->form_validation->set_rules('price', 'Price (USD)', 'required|numeric');
            $this->form_validation->set_rules('daily_cap', 'Daily Cap', 'required|integer');
            
            if ($this->form_validation->run()) {
                $update_data = array(
                    'offer_name' => $this->input->post('offer_name'),
                    'price' => $this->input->post('price'),
                    'daily_cap' => $this->input->post('daily_cap'),
                    'app_desc' => $this->input->post('app_desc'),
                    'preview_link' => $this->input->post('preview_link'),
                    'custom_payout' => $this->input->post('custom_payout'),
                    'platform' => $this->input->post('platform'),
                    'geo' => $this->input->post('geo'),
                    'app_category' => $this->input->post('app_category'),
                    'requirement_traffic' => $this->input->post('requirement_traffic'),
                    'icon_link' => $this->input->post('icon_link'),
                    'app_name' => $this->input->post('app_name'),
                    'app_size' => $this->input->post('app_size'),
                    'app_rate' => $this->input->post('app_rate'),
                    'app_id' => $this->input->post('app_id'),
                    'status' => $this->input->post('status'),
                    'is_enabled' => $this->input->post('is_enabled') ? 1 : 0
                );
                
                if ($this->M_hugoffers->updateOffer($id, $update_data)) {
                    $this->session->set_flashdata('message', 
                        '<div class="alert alert-success">Offer updated successfully!</div>');
                    redirect('admin/hugoffers');
                } else {
                    $this->session->set_flashdata('message', 
                        '<div class="alert alert-danger">Failed to update offer.</div>');
                }
            }
        }
        
        $this->render('hugoffers_edit', $this->data);
    }

    /**
     * Delete offer
     */
    public function delete($id = 0)
    {
        $offer = $this->M_hugoffers->getOfferById($id);
        if (!$offer) {
            show_404();
        }
        
        if ($this->M_hugoffers->deleteOffer($id)) {
            $this->session->set_flashdata('message', 
                '<div class="alert alert-success">Offer deleted successfully!</div>');
        } else {
            $this->session->set_flashdata('message', 
                '<div class="alert alert-danger">Failed to delete offer.</div>');
        }
        
        redirect('admin/hugoffers');
    }

    /**
     * Sync offer from API
     */
    public function sync($campid = '')
    {
        if (empty($campid)) {
            show_404();
        }
        
        $api_key = $this->data['settings']['hugoffers_api_key'];
        if (empty($api_key)) {
            $this->session->set_flashdata('message', 
                '<div class="alert alert-danger">API key not configured. Please set it in settings first.</div>');
            redirect('admin/hugoffers');
        }
        
        if ($this->M_hugoffers->syncOfferFromAPI($campid, $api_key)) {
            $this->session->set_flashdata('message', 
                '<div class="alert alert-success">Offer synced successfully from API!</div>');
        } else {
            $this->session->set_flashdata('message', 
                '<div class="alert alert-danger">Failed to sync offer. Campaign ID may not exist in API.</div>');
        }
        
        redirect('admin/hugoffers');
    }

    /**
     * Add offer by Campaign ID (quick add)
     */
    public function quick_add()
    {
        if ($this->input->post()) {
            $campid = trim($this->input->post('campid'));
            $custom_payout = $this->input->post('custom_payout');
            
            if (empty($campid)) {
                $this->session->set_flashdata('message', 
                    '<div class="alert alert-danger">Campaign ID is required.</div>');
                redirect('admin/hugoffers');
            }
            
            // Check if offer already exists
            if ($this->M_hugoffers->getOfferByCampId($campid)) {
                $this->session->set_flashdata('message', 
                    '<div class="alert alert-warning">Offer with Campaign ID ' . $campid . ' already exists.</div>');
                redirect('admin/hugoffers');
            }
            
            $api_key = $this->data['settings']['hugoffers_api_key'];
            
            if (!empty($api_key)) {
                // Try to sync from API first
                try {
                    if ($this->M_hugoffers->syncOfferFromAPI($campid, $api_key)) {
                        // Update custom payout if provided
                        if (!empty($custom_payout)) {
                            $offer = $this->M_hugoffers->getOfferByCampId($campid);
                            if ($offer) {
                                $this->M_hugoffers->updateOffer($offer['id'], array('custom_payout' => floatval($custom_payout)));
                            }
                        }
                        
                        $this->session->set_flashdata('message', 
                            '<div class="alert alert-success">Offer added and synced from API! Images and details fetched automatically.</div>');
                        redirect('admin/hugoffers');
                    }
                } catch (Exception $e) {
                    log_message('error', 'HugOffers API Sync Error: ' . $e->getMessage());
                    // Continue to manual add if API fails
                }
            }
            
            // If API sync failed, add manually with basic data
            $offer_data = array(
                'campid' => $campid,
                'offer_name' => 'Offer ' . $campid,
                'price' => !empty($custom_payout) ? floatval($custom_payout) : 1.0,
                'daily_cap' => 100,
                'custom_payout' => !empty($custom_payout) ? floatval($custom_payout) : null,
                'added_by' => $this->data['admin']['id']
            );
            
            if ($this->M_hugoffers->addOffer($offer_data)) {
                $this->session->set_flashdata('message', 
                    '<div class="alert alert-success">Offer added manually. Please edit to complete details.</div>');
            } else {
                $this->session->set_flashdata('message', 
                    '<div class="alert alert-danger">Failed to add offer.</div>');
            }
        }
        
        redirect('admin/hugoffers');
    }

    /**
     * Toggle offer status
     */
    public function toggle_status($id = 0)
    {
        $offer = $this->M_hugoffers->getOfferById($id);
        if (!$offer) {
            show_404();
        }
        
        $new_status = $offer['is_enabled'] ? 0 : 1;
        
        if ($this->M_hugoffers->updateOffer($id, array('is_enabled' => $new_status))) {
            $status_text = $new_status ? 'enabled' : 'disabled';
            $this->session->set_flashdata('message', 
                '<div class="alert alert-success">Offer ' . $status_text . ' successfully!</div>');
        } else {
            $this->session->set_flashdata('message', 
                '<div class="alert alert-danger">Failed to update offer status.</div>');
        }
        
        redirect('admin/hugoffers');
    }

    /**
     * Bulk import offers from list
     */
    public function bulk_import()
    {
        $this->data['page'] = 'Bulk Import HugOffers';
        
        if ($this->input->post()) {
            $offers_list = trim($this->input->post('offers_list'));
            
            if (empty($offers_list)) {
                $this->session->set_flashdata('message', 
                    '<div class="alert alert-danger">Please provide the offers list.</div>');
                redirect('admin/hugoffers/bulk_import');
            }
            
            $lines = explode("\n", $offers_list);
            $imported = 0;
            $failed = 0;
            
            // Split into offer blocks (separated by empty lines)
            $offer_blocks = preg_split('/\n\s*\n/', $offers_list);
            
            foreach ($offer_blocks as $block) {
                $block = trim($block);
                if (empty($block)) continue;
                
                $lines_in_block = explode("\n", $block);
                $campid = '';
                $name = '';
                $payout = 0.0;
                $caps = 100;
                $url = '';
                
                foreach ($lines_in_block as $line) {
                    $line = trim($line);
                    
                    // Parse [ID]Name format
                    if (preg_match('/\[(\d+)\](.+)/', $line, $matches)) {
                        $campid = $matches[1];
                        $name = trim($matches[2]);
                    }
                    // Parse URL
                    elseif (preg_match('/^https?:\/\//', $line)) {
                        $url = $line;
                    }
                    // Parse PO: $X.XX (with different colon types)
                    elseif (preg_match('/PO[：:]\s*\$?([0-9.]+)/i', $line, $payout_matches)) {
                        $payout = floatval($payout_matches[1]);
                    }
                    // Parse caps
                    elseif (preg_match('/caps?\s*:?\s*(\d+)/i', $line, $caps_matches)) {
                        $caps = intval($caps_matches[1]);
                    }
                }
                
                if (!empty($campid) && !empty($name) && $payout > 0) {
                    $offer_data = array(
                        'campid' => $campid,
                        'offer_name' => $name,
                        'price' => $payout,
                        'daily_cap' => $caps,
                        'preview_link' => $url,
                        'custom_payout' => $payout,
                        'added_by' => $this->data['admin']['id']
                    );
                    
                    if ($this->M_hugoffers->addOffer($offer_data)) {
                        $imported++;
                    } else {
                        $failed++;
                    }
                } else {
                    $failed++;
                }
            }
            
            $this->session->set_flashdata('message', 
                '<div class="alert alert-success">Import completed! Imported: ' . $imported . ', Failed: ' . $failed . '</div>');
            redirect('admin/hugoffers');
        }
        
        $this->render('hugoffers_bulk_import', $this->data);
    }

    /**
     * Test API connectivity and response
     */
    public function test_api()
    {
        $this->data['page'] = 'Test HugOffers API';
        
        $api_key = $this->data['settings']['hugoffers_api_key'];
        if (empty($api_key)) {
            $this->session->set_flashdata('message', 
                '<div class="alert alert-danger">API key not configured. Please set it in settings first.</div>');
            redirect('admin/hugoffers');
        }
        
        $this->load->helper('hugoffers_helper');
        
        // Test API response
        $api_response = get_hugoffers_api_data($api_key);
        
        $this->data['api_response'] = $api_response;
        $this->data['api_key'] = $api_key;
        
        // If we have offers, format one as example
        if (isset($api_response['offers']) && !empty($api_response['offers'])) {
            $this->data['sample_offer'] = format_hugoffers_offer($api_response['offers'][0]);
        }
        
        $this->render('hugoffers_test_api', $this->data);
    }
} 