<?php
defined('BASEPATH') or exit('No direct script access allowed');
include APPPATH . 'third_party/coinbase/autoload.php';

use CoinbaseCommerce\ApiClient;
use CoinbaseCommerce\Resources\Charge;
use CoinbaseCommerce\Webhook;

class Wh extends Guess_Controller
{
	public function __construct()
	{
		parent::__construct();
		$this->load->model(['m_deposit', 'm_offerwall']);
		$this->data['settings'] = $this->m_core->getSettings();
		$this->data['whitelist_ips'] = [
			'coinbase' => [],
			'faucetpay' => [],
			'wannads' => [
				'34.250.159.173',
				'34.244.210.150',
				'52.212.236.135',
				'34.251.83.149'
			],
			'adgatemedia' => [
                '52.42.57.125'
            ],
			'cpx' => [
				'157.90.97.92',
				'188.40.3.73',
				'2a01:4f8:d0a:30ff::2'	
			],
			'offertoro' => [
				'54.175.173.245'
					],
			'ayetstudios' => [
				'35.165.166.40',
				'35.166.159.131',
				'52.40.3.140'
			],
			'personaly' => [
				'159.203.84.146',
				'52.200.142.249'
			],
			'hangmyads' => [
				'52.10.12.101',
				'216.137.182.154'
			],
			'bitswall' => [
				'188.165.198.204',
				'2001:41d0:2:8fcc::'
			],
			'payeer' => [
				'185.71.65.92',
				'185.71.65.189',
				'149.202.17.210'
			],
			'adscendmedia' => [
				'54.204.57.82',
				'52.117.122.183',
				'52.177.127.192',
				'52.117.121.196'
          ]

		];
	}

	public function coinbase()
	{
		if ($this->data['settings']['coinbase_deposit_status'] == 'off') {
			die();
		}
		$secret = $this->data['settings']['coinbase_secret'];
		$signraturHeader = isset($_SERVER['HTTP_X_CC_WEBHOOK_SIGNATURE']) ? $_SERVER['HTTP_X_CC_WEBHOOK_SIGNATURE'] : null;
		$payload = trim(file_get_contents('php://input'));

		try {
			$event = Webhook::buildEvent($payload, $signraturHeader, $secret);
			http_response_code(200);
			switch ($event->type) {
				case 'charge:created':
					$this->m_deposit->updateStatus($event->data->code, 'Created');
					break;
				case 'charge:confirmed':
					$this->m_deposit->updateStatus($event->data->code, 'Confirmed');
					$this->m_deposit->depositSuccess($event->data->code);
					break;
				case 'charge:failed':
					$this->m_deposit->updateStatus($event->data->code, 'Failed');
					break;
				case 'charge:delayed':
					$this->m_deposit->updateStatus($event->data->code, 'Delayed');
					break;
				case 'charge:pending':
					$this->m_deposit->updateStatus($event->data->code, 'Pending');
					break;
				case 'charge:resolved':
					$this->m_deposit->updateStatus($event->data->code, 'Confirmed');
					$this->m_deposit->updateStatus($event->data->code, 'Resolved');
					break;
			}
			echo sprintf('Successully verified event with id %s and type %s.', $event->id, $event->type);
		} catch (\Exception $exception) {
			http_response_code(400);
			echo 'Error occured. ' . $exception->getMessage();
		}
	}

	public function faucetpay()
	{
		if ($this->data['settings']['faucetpay_deposit_status'] == 'off') {
			die();
		}
		$token = $this->input->post('token');
		$validate = @json_decode(get_data('https://faucetpay.io/merchant/get-payment/' . $token), TRUE);
		if ($validate['valid'] && $validate['merchant_username'] == $this->data['settings']['faucetpay_username'] && $validate['currency1'] == 'USDT' && $validate['amount1'] >= $this->data['settings']['faucetpay_min_deposit']) {
			if ($this->data['settings']['faucetpay_currency'] != "") {
				$faucetpayMethods = explode(',', $this->data['settings']['faucetpay_currency']);
				if (in_array($validate['currency2'], $faucetpayMethods)) {
					$this->m_deposit->addDeposit($validate['custom'], $validate['amount1'], $validate['transaction_id'], 1, 'Confirmed');
					$this->m_deposit->updateUser($validate['custom'], $validate['amount1']);
				}
			} else {
				$this->m_deposit->addDeposit($validate['custom'], $validate['amount1'], $validate['transaction_id'], 1, 'Confirmed');
				$this->m_deposit->updateUser($validate['custom'], $validate['amount1']);
			}
		}
	}

	public function payeer()
	{
		if (!in_array($this->input->ip_address(), $this->data['whitelist_ips']['payeer'])) {
			echo 'ok';
			die();
		}
		if (isset($_POST['m_operation_id']) && isset($_POST['m_sign'])) {
			$arHash = array(
				$_POST['m_operation_id'],
				$_POST['m_operation_ps'],
				$_POST['m_operation_date'],
				$_POST['m_operation_pay_date'],
				$_POST['m_shop'],
				$_POST['m_orderid'],
				$_POST['m_amount'],
				$_POST['m_curr'],
				$_POST['m_desc'],
				$_POST['m_status']
			);

			if (isset($_POST['m_params'])) {
				$arHash[] = $_POST['m_params'];
			}

			$arHash[] = $this->data['settings']['payeer_secret'];

			$sign_hash = strtoupper(hash('sha256', implode(':', $arHash)));

			if ($_POST['m_sign'] == $sign_hash && $_POST['m_status'] == 'success' && $_POST['m_amount'] >= $this->data['settings']['payeer_min_deposit']) {
				$orderId = $this->db->escape_str($_POST['m_orderid']);
				$this->m_deposit->updateStatus($orderId, 'Confirmed');
				$this->m_deposit->depositSuccess($orderId);
				die($_POST['m_orderid'] . '|success');
			}

			die($_POST['m_orderid'] . '|error');
		}
	}

	public function wannads()
	{
		if (!in_array($this->input->ip_address(), $this->data['whitelist_ips']['wannads'])) {
			die();
		}
		$userId = isset($_GET['subId']) ? $this->db->escape_str($_GET['subId']) : null;
		$transactionId = isset($_GET['transId']) ? $this->db->escape_str($_GET['transId']) : null;
		$reward = isset($_GET['reward']) ? $this->db->escape_str($_GET['reward']) : null;
		$signature = isset($_GET['signature']) ? $this->db->escape_str($_GET['signature']) : null;
		$action = isset($_GET['status']) ? $this->db->escape_str($_GET['status']) : null;
		$userIp = isset($_GET['userIp']) ? $this->db->escape_str($_GET['userIp']) : "0.0.0.0";

		if (md5($userId . $transactionId . $reward . $this->data['settings']['wannads_secret_key']) != $signature) {
			echo "ERROR: Signature doesn't match";
			return;
		}

		$reward = $reward * $this->data['settings']['currency_rate'];

		$trans = $this->m_offerwall->getTransaction($transactionId, 'wannads');
		if ($action == 2) {
			$this->m_offerwall->reduceUserBalance($userId, abs($reward));
			$this->m_offerwall->insertTransaction($userId, 'wannads', $userIp, $reward, $transactionId, 1, time());
			echo "OK";
		} else {
			if (!$trans) {
				$hold = 0;
				if ($reward > $this->data['settings']['offerwall_min_hold']) {
					$hold = $this->data['settings']['wannads_hold'];
				}
				if ($hold == 0) {
					$offerId = $this->m_offerwall->insertTransaction($userId, 'wannads', $userIp, $reward, $transactionId, 2, time());
					$this->m_offerwall->updateUserBalance($userId, $reward);
					$this->m_core->addNotification($userId, currencyDisplay($reward, $this->data['settings']) . " from Wannads Offer #" . $offerId . " was credited to your balance.", 1);

					$user = $this->m_core->getUserFromId($userId);
					$this->m_core->addExp($user['id'], $this->data['settings']['offerwall_exp_reward']);
					if (($user['exp'] + $this->data['settings']['offerwall_exp_reward']) >= ($user['level'] + 1) * 100) {
						$this->m_core->levelUp($user['id']);
					}
				} else {
					$availableAt = time() + $hold * 86400;
					$offerId = $this->m_offerwall->insertTransaction($userId, 'wannads', $userIp, $reward, $transactionId, 0, $availableAt);
					$this->m_core->addNotification($userId, "Your Wannads Offer #" . $offerId . " is pending approval.", 0);
				}
				echo "OK";
			} else {
				echo "DUP";
			}
		}
	}

	public function bonanzawall()
  {
    $secret = $this->data['settings']['bonanzawall_secret'];
    $userId = isset($_REQUEST['sub_id']) ? $this->db->escape_str($_REQUEST['sub_id']) : null;
    $transactionId = isset($_REQUEST['trans_id']) ? $this->db->escape_str($_REQUEST['trans_id']) : null;
    $reward = isset($_REQUEST['amount']) ? $this->db->escape_str($_REQUEST['amount']) : null;
    $userIp = isset($_REQUEST['ip_address']) ? $this->db->escape_str($_REQUEST['ip_address']) : "0.0.0.0";
    $signature = isset($_REQUEST['signature']) ? $this->db->escape_str($_REQUEST['signature']) : null;
    if (md5($userId . $transactionId . $reward . $secret) != $signature) {
      echo "ERROR: Signature doesn't match";
      return;
    }

    $amountraw = $reward * $this->data['settings']['currency_rate'];

    $trans = $this->m_offerwall->getTransaction($transactionId, 'BonanzaWall');
    if (!$trans) {
      $hold = 0;
      if ($amountraw > $this->data['settings']['offerwall_min_hold']) {
        $hold = $this->data['settings']['bonanzawall_hold'];
      }
      if ($hold == 0) {
        $offerId = $this->m_offerwall->insertTransaction($userId, 'BonanzaWall', $userIp, $amountraw, $transactionId, 2, time());
        $this->m_offerwall->updateUserBalance($userId, $amountraw);
        $this->m_core->addNotification($userId, currencyDisplay($amountraw, $this->data['settings']) . " from BonanzaWall Offer #" . $offerId . " was credited to your balance.", 1);

        $user = $this->m_core->getUserFromId($userId);
        $this->m_core->addEnergy($user['id'], $this->data['settings']['offerwall_energy']); 
        $this->m_core->addExp($user['id'], $this->data['settings']['offerwall_exp_reward']);
        if (($user['exp'] + $this->data['settings']['offerwall_exp_reward']) >= ($user['level'] + 1) * 100) {
          $this->m_core->levelUp($user['id']);
        }
      } else {
        $availableAt = time() + $hold * 86400;
        $offerId = $this->m_offerwall->insertTransaction($userId, 'BonanzaWall', $userIp, $amountraw, $transactionId, 0, $availableAt);
        $this->m_core->addNotification($userId, "Your BonanzaWall Offer #" . $offerId . " is pending approval.", 0);
      }
      echo "1";
    } else {
      echo "0";
    }
  }
	public function offertoro()
	{
		if (!in_array($this->input->ip_address(), $this->data['whitelist_ips']['offertoro'])) {
			echo 'ok';
			die();
		}
		$secret = $this->data['settings']['offertoro_app_secret'];

		$userId = isset($_GET['user_id']) ? $this->db->escape_str($_GET['user_id']) : 2;
		$transactionId = isset($_GET['oid']) ? $this->db->escape_str($_GET['oid']) : null;
		$offerId = isset($_GET['oid']) ? $this->db->escape_str($_GET['oid']) : null;
		$reward = isset($_GET['amount']) ? $this->db->escape_str($_GET['amount']) : null;
		$ipAddress = isset($_GET['ip_address']) ? $this->db->escape_str($_GET['ip_address']) : null;
		$signature = isset($_GET['sig']) ? $this->db->escape_str($_GET['sig']) : null;

		if (md5($offerId . '-' . $userId . '-' . $secret) != $signature) {
			echo 0;
			return;
		}

		$reward = $reward * $this->data['settings']['currency_rate'];

		if ($reward < 0) {
			$this->m_offerwall->reduceUserBalance($userId, abs($reward));
			$this->m_offerwall->insertTransaction($userId, 'Offertoro', $ipAddress, $reward, $transactionId, 1, time());
			echo 1;
		} else {
			$trans = $this->m_offerwall->getTransaction($transactionId, 'offertoro');
			if (!$trans) {
				$hold = 0;
				if ($reward > $this->data['settings']['offerwall_min_hold']) {
					$hold = $this->data['settings']['offertoro_hold'];
				}
				if ($hold == 0) {
					$offerId = $this->m_offerwall->insertTransaction($userId, 'offertoro', $ipAddress, $reward, $transactionId, 2, time());
					$this->m_offerwall->updateUserBalance($userId, $reward);
					$this->m_core->addNotification($userId, currencyDisplay($reward, $this->data['settings']) . " from Offertoro Offer #" . $offerId . " was credited to your balance.", 1);

					$user = $this->m_core->getUserFromId($userId);
					$this->m_core->addExp($user['id'], $this->data['settings']['offerwall_exp_reward']);
					if (($user['exp'] + $this->data['settings']['offerwall_exp_reward']) >= ($user['level'] + 1) * 100) {
						$this->m_core->levelUp($user['id']);
					}
				} else {
					$availableAt = time() + $hold * 86400;
					$offerId = $this->m_offerwall->insertTransaction($userId, 'offertoro', $ipAddress, $reward, $transactionId, 0, $availableAt);
					$this->m_core->addNotification($userId, "Your Offertoro Offer #" . $offerId . " is pending approval.", 0);
				}
				echo 1;
			} else {
				echo 1;
			}
		}
	}

	public function cpx()
	{
		if (!in_array($this->input->ip_address(), $this->data['whitelist_ips']['cpx'])) {
			echo 'ok';
			die();
		}
		$secret = $this->data['settings']['cpx_hash'];
		$userId = isset($_GET['user_id']) ? $this->db->escape_str($_GET['user_id']) : null;
		$action = isset($_GET['status']) ? $this->db->escape_str($_GET['status']) : null;
		$transactionId = isset($_GET['trans_id']) ? $this->db->escape_str($_GET['trans_id']) : null;
		$reward = isset($_GET['amount']) ? $this->db->escape_str($_GET['amount']) : null;
		$userIp = isset($_GET['ip_click']) ? $this->db->escape_str($_GET['ip_click']) : "0.0.0.0";
		$signature = isset($_GET['hash']) ? $this->db->escape_str($_GET['hash']) : null;

		if (md5($transactionId . '-' . $secret) != $signature) {
			echo "ERROR: Signature doesn't match";
			return;
		}
        $reward = $reward * $this->data['settings']['currency_rate'];

// Then, check if Happy Hour is ON and apply 50% extra
if ($this->data['settings']['happyhour_status'] == 'on') {
    $reward *= 1.50;  // Add 50% extra to the currency amount
}
		$trans = $this->m_offerwall->getTransaction($transactionId, 'CPX Research');
		if ($action == 2) {
			$this->m_offerwall->reduceUserBalance($userId, abs($reward));
			$this->m_offerwall->insertTransaction($userId, 'CPX Research', $userIp, $reward, $transactionId, 1, time());
			echo "OK";
		} else {
			if (!$trans) {
				$hold = 0;
				if ($reward > $this->data['settings']['offerwall_min_hold']) {
					$hold = $this->data['settings']['cpx_hold'];
				}
				if ($hold == 0) {
					$offerId = $this->m_offerwall->insertTransaction($userId, 'CPX Research', $userIp, $reward, $transactionId, 2, time());
					$this->m_offerwall->updateUserBalance($userId, $reward);
					$this->m_core->addNotification($userId, currencyDisplay($reward, $this->data['settings']) . " from CPX Research Offer #" . $offerId . " was credited to your balance.", 1);

					$user = $this->m_core->getUserFromId($userId);
					$this->m_core->addExp($user['id'], $this->data['settings']['offerwall_exp_reward']);
					if (($user['exp'] + $this->data['settings']['offerwall_exp_reward']) >= ($user['level'] + 1) * 100) {
						$this->m_core->levelUp($user['id']);
					}
				} else {
					$availableAt = time() + $hold * 86400;
					$offerId = $this->m_offerwall->insertTransaction($userId, 'CPX Research', $userIp, $reward, $transactionId, 0, $availableAt);
					$this->m_core->addNotification($userId, "Your CPX Research Offer #" . $offerId . " is pending approval.", 0);
				}
				echo "OK";
			} else {
				echo "DUP";
			}
		}
	}

	public function ayetstudios()
	{
		$userId = isset($_REQUEST['uid']) ? $this->db->escape_str($_REQUEST['uid']) : null;
		$transactionId = isset($_REQUEST['transaction_id']) ? $this->db->escape_str($_REQUEST['transaction_id']) : null;
		$action = isset($_REQUEST['is_chargeback']) ? $this->db->escape_str($_REQUEST['is_chargeback']) : null;
		$reward = isset($_REQUEST['currency_amount']) ? $this->db->escape_str($_REQUEST['currency_amount']) : null;
		$userIp = isset($_REQUEST['ip']) ? $this->db->escape_str($_REQUEST['ip']) : "not available";
		$signature = isset($_SERVER['HTTP_X_AYETSTUDIOS_SECURITY_HASH']) ? $this->db->escape_str($_SERVER['HTTP_X_AYETSTUDIOS_SECURITY_HASH']) : null;

		ksort($_REQUEST, SORT_STRING);
		$sortedQueryString = http_build_query($_REQUEST, '', '&');
		$securityHash = hash_hmac('sha256', $sortedQueryString, $this->data['settings']['ayetstudios_api']);
		error_log($_SERVER['HTTP_X_AYETSTUDIOS_SECURITY_HASH']);
		if ($securityHash != $signature) {
			echo "invalid signature";
			return;
		}
		$reward = $reward * $this->data['settings']['currency_rate'];

		$trans = $this->m_offerwall->getTransaction($transactionId, 'AyetStudios');
		if ($action == 1) {
			$this->m_offerwall->reduceUserBalance($userId, abs($reward));
			$this->m_offerwall->insertTransaction($userId, 'AyetStudios', $userIp, $reward, $transactionId, 1, time());
			echo "ok";
		} else {
			if (!$trans) {
				$hold = 0;
				if ($reward > $this->data['settings']['offerwall_min_hold']) {
					$hold = $this->data['settings']['ayetstudios_hold'];
				}
				if ($hold == 0) {
					$offerId = $this->m_offerwall->insertTransaction($userId, 'AyetStudios', $userIp, $reward, $transactionId, 2, time());
					$this->m_offerwall->updateUserBalance($userId, $reward);
					$this->m_core->addNotification($userId, currencyDisplay($reward, $this->data['settings']) . " from AyetStudios Offer #" . $offerId . " was credited to your balance.", 1);

					$user = $this->m_core->getUserFromId($userId);
					$this->m_core->addExp($user['id'], $this->data['settings']['offerwall_exp_reward']);
					if (($user['exp'] + $this->data['settings']['offerwall_exp_reward']) >= ($user['level'] + 1) * 100) {
						$this->m_core->levelUp($user['id']);
					}
				} else {
					$availableAt = time() + $hold * 86400;
					$offerId = $this->m_offerwall->insertTransaction($userId, 'AyetStudios', $userIp, $reward, $transactionId, 0, $availableAt);
					$this->m_core->addNotification($userId, "Your AyetStudios Offer #" . $offerId . " is pending approval.", 0);
				}
				echo "ok";
			} else {
				echo "ok";
			}
		}
	}
	public function offerdaddy()
	{
		$transactionId = $this->db->escape_str(urldecode($_GET["transaction_id"]));
		$offer_id = $this->db->escape_str(urldecode($_GET["offer_id"]));
		$reward = $this->db->escape_str(urldecode($_GET["amount"]));
		$userId = $this->db->escape_str(urldecode($_GET["userid"]));
		$signature = urldecode($_GET["signature"]);

		//Check the signature
		$validationSignature = md5($transactionId . "/" . $offer_id . "/" . $this->data['settings']['offerdaddy_app_key']);

		if ($validationSignature != trim($signature)) {
			echo "0";
			die();
		}
		$reward = $reward * $this->data['settings']['currency_rate'];

		$trans = $this->m_offerwall->getTransaction($transactionId, 'OfferDady');
		if ($reward < 0) {
			$this->m_offerwall->reduceUserBalance($userId, abs($reward));
			$this->m_offerwall->insertTransaction($userId, 'OfferDady', 'not available', $reward, $transactionId, 1, time());
			echo "1";
		} else {
			if (!$trans) {
				$hold = 0;
				if ($reward > $this->data['settings']['offerwall_min_hold']) {
					$hold = $this->data['settings']['offerdaddy_hold'];
				}
				if ($hold == 0) {
					$offerId = $this->m_offerwall->insertTransaction($userId, 'OfferDady', 'not available', $reward, $transactionId, 2, time());
					$this->m_offerwall->updateUserBalance($userId, $reward);
					$this->m_core->addNotification($userId, currencyDisplay($reward, $this->data['settings']) . " from OfferDady Offer #" . $offerId . " was credited to your balance.", 1);

					$user = $this->m_core->getUserFromId($userId);
					$this->m_core->addExp($user['id'], $this->data['settings']['offerwall_exp_reward']);
					if (($user['exp'] + $this->data['settings']['offerwall_exp_reward']) >= ($user['level'] + 1) * 100) {
						$this->m_core->levelUp($user['id']);
					}
				} else {
					$availableAt = time() + $hold * 86400;
					$offerId = $this->m_offerwall->insertTransaction($userId, 'OfferDady', 'not available', $reward, $transactionId, 0, $availableAt);
					$this->m_core->addNotification($userId, "Your OfferDady Offer #" . $offerId . " is pending approval.", 0);
				}
				echo "1";
			} else {
				echo "1";
			}
		}
	}
	public function personaly()
	{
		if (!in_array($this->input->ip_address(), $this->data['whitelist_ips']['personaly'])) {
			echo 'ok';
			die();
		}
		$transactionId = isset($_GET['offer_id']) ? $this->db->escape_str($_GET['offer_id']) : null;
		$reward = isset($_GET['amount']) ? $this->db->escape_str($_GET['amount']) : null;
		$userId = isset($_GET['user_id']) ? $this->db->escape_str($_GET['user_id']) : null;
		$userIp = isset($_GET['user_ip']) ? $this->db->escape_str($_GET['user_ip']) : "not available";
		$signature = isset($_GET['signature']) ? $this->db->escape_str($_GET['signature']) : "null";

		//Check the signature
		$validationSignature = md5($userId . ':' . $this->data['settings']['personaly_hash'] . ':' . $this->data['settings']['personaly_secret_key']);

		if ($validationSignature != trim($signature)) {
			echo "0";
			die();
		}
		$reward = $reward * $this->data['settings']['currency_rate'];

		if ($reward < 0) {
			$this->m_offerwall->reduceUserBalance($userId, abs($reward));
			$this->m_offerwall->insertTransaction($userId, 'Persona.ly', $userIp, $reward, $transactionId, 1, time());
			echo "1";
		} else {
			$hold = 0;
			if ($reward > $this->data['settings']['offerwall_min_hold']) {
				$hold = $this->data['settings']['personaly_hold'];
			}
			if ($hold == 0) {
				$offerId = $this->m_offerwall->insertTransaction($userId, 'Persona.ly', $userIp, $reward, $transactionId, 2, time());
				$this->m_offerwall->updateUserBalance($userId, $reward);
				$this->m_core->addNotification($userId, currencyDisplay($reward, $this->data['settings']) . " from Persona.ly Offer #" . $offerId . " was credited to your balance.", 1);

				$user = $this->m_core->getUserFromId($userId);
				$this->m_core->addExp($user['id'], $this->data['settings']['offerwall_exp_reward']);
				if (($user['exp'] + $this->data['settings']['offerwall_exp_reward']) >= ($user['level'] + 1) * 100) {
					$this->m_core->levelUp($user['id']);
				}
			} else {
				$availableAt = time() + $hold * 86400;
				$offerId = $this->m_offerwall->insertTransaction($userId, 'Persona.ly', $userIp, $reward, $transactionId, 0, $availableAt);
				$this->m_core->addNotification($userId, "Your Persona.ly Offer #" . $offerId . " is pending approval.", 0);
			}
			echo "1";
		}
	}




     
	public function offers4crypto()
	{

		$userId = isset($_REQUEST['subId']) ? $this->db->escape_str($_REQUEST['subId']) : null;
		$transactionId = isset($_REQUEST['transId']) ? $this->db->escape_str($_REQUEST['transId']) : null;
		$reward = isset($_REQUEST['reward']) ? $this->db->escape_str($_REQUEST['reward']) : null;
		$payout = isset($_REQUEST['payout']) ? $this->db->escape_str($_REQUEST['payout']) : null;
		$userIp = isset($_REQUEST['userIp']) ? $this->db->escape_str($_REQUEST['userIp']) : "0.0.0.0";
		$signature = isset($_REQUEST['signature']) ? $this->db->escape_str($_REQUEST['signature']) : null;
		if (md5($userId . $transactionId . $reward . $this->data['settings']['offers4crypto_secret']) != $signature) {
			echo "ERROR: Signature doesn't match";
			return;
		}
		$reward = $reward * $this->data['settings']['currency_rate'];
// Then, check if Happy Hour is ON and apply 50% extra
if ($this->data['settings']['happyhour_status'] == 'on') {
    $reward *= 1.50;  // Add 50% extra to the currency amount
}


		$trans = $this->m_offerwall->getTransaction($transactionId, 'Offerwall.me');
		if (!$trans) {
			$hold = 0;
			if ($reward > $this->data['settings']['offerwall_min_hold']) {
				$hold = $this->data['settings']['offers4crypto_hold'];
			}
			if ($hold == 0) {
				$offerId = $this->m_offerwall->insertTransaction($userId, 'Offerwall.me', $userIp, $reward, $transactionId, 2, time(), '', $payout);
				$this->m_offerwall->updateUserBalance($userId, $reward);
				$this->m_core->addNotification($userId, currencyDisplay($reward, $this->data['settings']) . " from Offerwall.me Offer #" . $offerId . " was credited to your balance.", 1);

				$user = $this->m_core->getUserFromId($userId);
				$this->m_core->addExp($user['id'], $this->data['settings']['offerwall_exp_reward']);
				if (($user['exp'] + $this->data['settings']['offerwall_exp_reward']) >= ($user['level'] + 1) * 100) {
					$this->m_core->levelUp($user['id']);
				}
			} else {
				$availableAt = time() + $hold * 86400;
				$offerId = $this->m_offerwall->insertTransaction($userId, 'Offerwall.me', $userIp, $reward, $transactionId, 0, $availableAt, '', $payout);
				$this->m_core->addNotification($userId, "Your Offerwall.me Offer #" . $offerId . " is pending approval.", 0);
			}
			echo "ok";
		} else {
			echo "DUP";
		}
	}
     
     public function adbreak()
	{
	    //https://coinzi.net/wh/adbreak?subId=[YOUR_USER_ID]&reward=[REWARD_VALUE]&offerName=[OFFER_NAME]&userIp=[USER_IP]&transId=[TXID]&payout=[PAYOUT]
		$userId = isset($_REQUEST['subId']) ? $this->db->escape_str($_REQUEST['subId']) : null;
		$transactionId = isset($_REQUEST['transId']) ? $this->db->escape_str($_REQUEST['transId']) : null;
		$reward = isset($_REQUEST['reward']) ? $this->db->escape_str($_REQUEST['reward']) : null;
		$payout = isset($_REQUEST['payout']) ? $this->db->escape_str($_REQUEST['payout']) : null;
		$offername = isset($_REQUEST['offerName']) ? $this->db->escape_str($_REQUEST['offerName']) : null;
		$userIp = isset($_REQUEST['userIp']) ? $this->db->escape_str($_REQUEST['userIp']) : "0.0.0.0";
		$reward = $reward * $this->data['settings']['currency_rate'];

		$trans = $this->m_offerwall->getTransaction($transactionId, 'AdbreakMedia');
		if (!$trans) {
			$hold = 0;
			if ($reward > $this->data['settings']['offerwall_min_hold']) {
				$hold = $this->data['settings']['break_hold'];
			}
			if ($hold == 0) {
				$offerId = $this->m_offerwall->insertTransaction($userId, 'AdbreakMedia', $userIp, $reward, $transactionId, 2, time(), '', $payout);
				$this->m_offerwall->updateUserBalance($userId, $reward);
				$this->m_core->addNotification($userId, currencyDisplay($reward, $this->data['settings']) . " from AdbreakMedia Offer #" . $offername . " was credited to your balance.", 1);

				$user = $this->m_core->getUserFromId($userId);
				$this->m_core->addExp($user['id'], $this->data['settings']['offerwall_exp_reward']);
				if (($user['exp'] + $this->data['settings']['offerwall_exp_reward']) >= ($user['level'] + 1) * 100) {
					$this->m_core->levelUp($user['id']);
				}
			} else {
				$availableAt = time() + $hold * 86400;
				$offerId = $this->m_offerwall->insertTransaction($userId, 'AdbreakMedia', $userIp, $reward, $transactionId, 0, $availableAt, '', $payout);
				$this->m_core->addNotification($userId, "Your AdbreakMedia Offer #" . $offername . " is pending approval.", 0);
			}
        echo "Approved";
		} else {
			echo "Unauthorized";
		}
	}

	public function revlum()
	{

		$userId = isset($_REQUEST['subId']) ? $this->db->escape_str($_REQUEST['subId']) : null;
		$transactionId = isset($_REQUEST['transId']) ? $this->db->escape_str($_REQUEST['transId']) : null;
		$reward = isset($_REQUEST['reward']) ? $this->db->escape_str($_REQUEST['reward']) : null;
		$userIp = isset($_REQUEST['userIp']) ? $this->db->escape_str($_REQUEST['userIp']) : "0.0.0.0";
		$signature = isset($_REQUEST['signature']) ? $this->db->escape_str($_REQUEST['signature']) : null;
		if (md5($userId . $transactionId . $reward . $this->data['settings']['revlum_secret']) != $signature) {
			echo "ERROR: Signature doesn't match";
			return;
		}
		$reward = $reward * $this->data['settings']['currency_rate'];

		$trans = $this->m_offerwall->getTransaction($transactionId, 'Revlum');
		if (!$trans) {
			$hold = 0;
			if ($reward > $this->data['settings']['offerwall_min_hold']) {
				$hold = $this->data['settings']['revlum_hold'];
			}
			if ($hold == 0) {
				$offerId = $this->m_offerwall->insertTransaction($userId, 'Revlum', $userIp, $reward, $transactionId, 2, time());
				$this->m_offerwall->updateUserBalance($userId, $reward);
				$this->m_core->addNotification($userId, currencyDisplay($reward, $this->data['settings']) . " from Revlum Offer #" . $offerId . " was credited to your balance.", 1);

				$user = $this->m_core->getUserFromId($userId);
				$this->m_core->addExp($user['id'], $this->data['settings']['offerwall_exp_reward']);
				if (($user['exp'] + $this->data['settings']['offerwall_exp_reward']) >= ($user['level'] + 1) * 100) {
					$this->m_core->levelUp($user['id']);
				}
			} else {
				$availableAt = time() + $hold * 86400;
				$offerId = $this->m_offerwall->insertTransaction($userId, 'Revlum', $userIp, $reward, $transactionId, 0, $availableAt);
				$this->m_core->addNotification($userId, "Your Revlum Offer #" . $offerId . " is pending approval.", 0);
			}
			echo "ok";
		} else {
			echo "DUP";
		}
	}

	public function excentiv()
	{

      $userId = isset($_GET['subId']) ? $this->db->escape_str($_GET['subId']) : null;
      $transactionId = isset($_GET['transId']) ? $this->db->escape_str($_GET['transId']) : null;
      $reward = isset($_GET['reward']) ? $this->db->escape_str($_GET['reward']) : null;
      $payout = isset($_GET['payout']) ? $this->db->escape_str($_GET['payout']) : null;
      $userIp = isset($_GET['userIp']) ? $this->db->escape_str($_GET['userIp']) : "0.0.0.0";
      $signature = isset($_GET['signature']) ? $this->db->escape_str($_GET['signature']) : null;
      $status = isset($_GET['status']) ? $this->db->escape_str($_GET['status']) : null;
		if (md5($userId . $transactionId . $reward . $this->data['settings']['excentiv_secret']) != $signature) {
			echo "ERROR: Signature doesn't match";
			return;
		}
		$reward = $reward * $this->data['settings']['currency_rate'];
// Then, check if Happy Hour is ON and apply 50% extra
if ($this->data['settings']['happyhour_status'] == 'on') {
    $reward *= 1.50;  // Add 50% extra to the currency amount
}

		$trans = $this->m_offerwall->getTransaction($transactionId, 'Excentiv');
		if (!$trans) {
			$hold = 0;
			if ($reward > $this->data['settings']['offerwall_min_hold']) {
				$hold = $this->data['settings']['excentiv_hold'];
			}
			if ($hold == 0) {
				$offerId = $this->m_offerwall->insertTransaction($userId, 'Excentiv', $userIp, $reward, $transactionId, 2, time(), '', $payout);
				$this->m_offerwall->updateUserBalance($userId, $reward);
				$this->m_core->addNotification($userId, currencyDisplay($reward, $this->data['settings']) . " from Excentiv Offer #" . $offerId . " was credited to your balance.", 1);

				$user = $this->m_core->getUserFromId($userId);
				$this->m_core->addExp($user['id'], $this->data['settings']['offerwall_exp_reward']);
				if (($user['exp'] + $this->data['settings']['offerwall_exp_reward']) >= ($user['level'] + 1) * 100) {
					$this->m_core->levelUp($user['id']);
				}
			} else {
				$availableAt = time() + $hold * 86400;
				$offerId = $this->m_offerwall->insertTransaction($userId, 'Excentiv', $userIp, $reward, $transactionId, 0, $availableAt, '', $payout);
				$this->m_core->addNotification($userId, "Your Excentiv Offer #" . $offerId . " is pending approval.", 0);
			}
			echo "ok";
		} else {
			echo "DUP";
		}
	}
	
	public function wallarcade()
	{

		$userId = isset($_REQUEST['subId']) ? $this->db->escape_str($_REQUEST['subId']) : null;
		$transactionId = isset($_REQUEST['transId']) ? $this->db->escape_str($_REQUEST['transId']) : null;
		$reward = isset($_REQUEST['reward']) ? $this->db->escape_str($_REQUEST['reward']) : null;
		$userIp = isset($_REQUEST['userIp']) ? $this->db->escape_str($_REQUEST['userIp']) : "0.0.0.0";
		$signature = isset($_REQUEST['signature']) ? $this->db->escape_str($_REQUEST['signature']) : null;
		if (md5($userId . $transactionId . $reward . $this->data['settings']['wallarcade_secret']) != $signature) {
			echo "ERROR: Signature doesn't match";
			return;
		}
		$reward = $reward * $this->data['settings']['currency_rate'];

		$trans = $this->m_offerwall->getTransaction($transactionId, 'WallArcade');
		if (!$trans) {
			$hold = 0;
			if ($reward > $this->data['settings']['offerwall_min_hold']) {
				$hold = $this->data['settings']['wallarcade_hold'];
			}
			if ($hold == 0) {
				$offerId = $this->m_offerwall->insertTransaction($userId, 'WallArcade', $userIp, $reward, $transactionId, 2, time());
				$this->m_offerwall->updateUserBalance($userId, $reward);
				$this->m_core->addNotification($userId, currencyDisplay($reward, $this->data['settings']) . " from WallArcade Offer #" . $offerId . " was credited to your balance.", 1);

				$user = $this->m_core->getUserFromId($userId);
				$this->m_core->addExp($user['id'], $this->data['settings']['offerwall_exp_reward']);
				if (($user['exp'] + $this->data['settings']['offerwall_exp_reward']) >= ($user['level'] + 1) * 100) {
					$this->m_core->levelUp($user['id']);
				}
			} else {
				$availableAt = time() + $hold * 86400;
				$offerId = $this->m_offerwall->insertTransaction($userId, 'WallArcade', $userIp, $reward, $transactionId, 0, $availableAt);
				$this->m_core->addNotification($userId, "Your WallArcade Offer #" . $offerId . " is pending approval.", 0);
			}
			echo "ok";
		} else {
			echo "DUP";
		}
	}
	
	public function cryptowall()
	  {
             $secret = ""; //UPDATE YOUR SECRET HERE   
             $userId = isset($_POST['userId']) ? $this->db->escape_str($_POST['userId']) : null;
             $transactionId = isset($_POST['transId']) ? $this->db->escape_str($_POST['transId']) : null;
             $reward = isset($_POST['amount']) ? $this->db->escape_str($_POST['amount']) : null;
             $action = isset($_POST['status']) ? $this->db->escape_str($_POST['status']) : null;
             $userIp = isset($_POST['userIp']) ? $this->db->escape_str($_POST['userIp']) : "0.0.0.0";
             $signature = isset($_POST['signature']) ? $this->db->escape_str($_POST['signature']) : null;
             if (md5($userId . $transactionId . $reward . $secret) != $signature) {
             echo "ERROR: Signature doesn't match";
             return;
             }
            
             $reward = $reward * $this->data['settings']['currency_rate'];
            
		$trans = $this->m_offerwall->getTransaction($transactionId, 'CryptoWall');
		if (!$trans) {
			$hold = 0;
			if ($reward > $this->data['settings']['offerwall_min_hold']) {
				$hold = 0; //UPDATE YOUR HOLD DAYS HERE
			}
			if ($hold == 0) {
				$offerId = $this->m_offerwall->insertTransaction($userId, 'CryptoWall', $userIp, $reward, $transactionId, 2, time());
				$this->m_offerwall->updateUserBalance($userId, $reward);
				$this->m_core->addNotification($userId, currencyDisplay($reward, $this->data['settings']) . " from CryptoWall Offer #" . $offerId . " was credited to your balance.", 1);

				$user = $this->m_core->getUserFromId($userId);
				$this->m_core->addExp($user['id'], $this->data['settings']['offerwall_exp_reward']);
				if (($user['exp'] + $this->data['settings']['offerwall_exp_reward']) >= ($user['level'] + 1) * 100) {
					$this->m_core->levelUp($user['id']);
				}
			} else {
				$availableAt = time() + $hold * 86400;
				$offerId = $this->m_offerwall->insertTransaction($userId, 'CryptoWall', $userIp, $reward, $transactionId, 0, $availableAt);
				$this->m_core->addNotification($userId, "Your CryptoWall Offer #" . $offerId . " is pending approval.", 0);
			}
			echo "ok";
		} else {
			echo "DUP";
		}
	}
		
public function adscend()
  {
    if (!in_array($this->input->ip_address(), $this->data['whitelist_ips']['adscendmedia'])) {
      echo 'ok';
      die();
    }
    $secret = $this->data['settings']['adscendmedia_secret'];

    $userId = isset($_GET['sub1']) ? $this->db->escape_str($_GET['sub1']) : null;
    $time = isset($_GET['time']) ? $this->db->escape_str($_GET['time']) : null;
    $transactionId = isset($_GET['transid']) ? $this->db->escape_str($_GET['transid']) : null;
    $offerId = isset($_GET['offerid']) ? $this->db->escape_str($_GET['offerid']) : null;
    $reward = isset($_GET['rate']) ? $this->db->escape_str($_GET['rate']) : null;
    $ipAddress = isset($_GET['ip_address']) ? $this->db->escape_str($_GET['ip_address']) : null;
    $signature = isset($_GET['hash']) ? $this->db->escape_str($_GET['hash']) : null;

    if (hash_hmac("md5", "sub1=" . $userId . "&time=" . $time . "&transid=" . $transactionId . "&offerid=" . $offerId . "&rate=" . $reward . "&ip_address=" . $ipAddress, $secret) != $signature) {
      echo 0;
      return;
       }

    $reward = $reward * $this->data['settings']['currency_rate'];

    if ($reward < 0) {
      $this->m_offerwall->reduceUserBalance($userId, abs($reward));
      $this->m_offerwall->insertTransaction($userId, 'Adscend Media', $ipAddress, $reward, $transactionId, 1, time());
      echo 1;
    } else {
      $trans = $this->m_offerwall->getTransaction($transactionId, 'Adscend');
      if (!$trans) {
        $hold = 0;
        if ($reward > $this->data['settings']['offerwall_min_hold']) {
          $hold = 10;
        }
        if ($hold == 0) {
          $offerId = $this->m_offerwall->insertTransaction($userId, 'Adscend', $ipAddress, $reward, $transactionId, 2, time());
          $this->m_offerwall->updateUserBalance($userId, $reward);
          $this->m_core->addNotification($userId, currencyDisplay($reward, $this->data['settings']) . " from Adscend Media Offer #" . $offerId . " was credited to your balance.", 1);

          $user = $this->m_core->getUserFromId($userId);
          $this->m_core->addExp($user['id'], $this->data['settings']['offerwall_exp_reward']);
          if (($user['exp'] + $this->data['settings']['offerwall_exp_reward']) >= ($user['level'] + 1) * 100) {
            $this->m_core->levelUp($user['id']);
          }
        } else {
          $availableAt = time() + $hold * 86400;
          $offerId = $this->m_offerwall->insertTransaction($userId, 'Adscend', $ipAddress, $reward, $transactionId, 0, $availableAt);
          $this->m_core->addNotification($userId, "Your Adscend Media Offer #" . $offerId . " is pending approval.", 0);
        }
        echo 1;
      } else {
        echo 1;
      }
    }
  }

	public function dripoffers()
	{

		$userId = isset($_REQUEST['subId']) ? $this->db->escape_str($_REQUEST['subId']) : null;
		$transactionId = isset($_REQUEST['transId']) ? $this->db->escape_str($_REQUEST['transId']) : null;
		$reward = isset($_REQUEST['reward']) ? $this->db->escape_str($_REQUEST['reward']) : null;
		$userIp = isset($_REQUEST['userIp']) ? $this->db->escape_str($_REQUEST['userIp']) : "0.0.0.0";
		$signature = isset($_REQUEST['signature']) ? $this->db->escape_str($_REQUEST['signature']) : null;
		if (md5($userId . $transactionId . $reward . $this->data['settings']['dripoffers_secret']) != $signature) {
			echo "ERROR: Signature doesn't match";
			return;
		}
		$reward = $reward * $this->data['settings']['currency_rate'];

		$trans = $this->m_offerwall->getTransaction($transactionId, 'DripOffers');
		if (!$trans) {
			$hold = 0;
			if ($reward > $this->data['settings']['offerwall_min_hold']) {
				$hold = $this->data['settings']['dripoffers_hold'];
			}
			if ($hold == 0) {
				$offerId = $this->m_offerwall->insertTransaction($userId, 'DripOffers', $userIp, $reward, $transactionId, 2, time());
				$this->m_offerwall->updateUserBalance($userId, $reward);
				$this->m_core->addNotification($userId, currencyDisplay($reward, $this->data['settings']) . " from DripOffers Offer #" . $offerId . " was credited to your balance.", 1);

				$user = $this->m_core->getUserFromId($userId);
				$this->m_core->addExp($user['id'], $this->data['settings']['offerwall_exp_reward']);
				if (($user['exp'] + $this->data['settings']['offerwall_exp_reward']) >= ($user['level'] + 1) * 100) {
					$this->m_core->levelUp($user['id']);
				}
			} else {
				$availableAt = time() + $hold * 86400;
				$offerId = $this->m_offerwall->insertTransaction($userId, 'DripOffers', $userIp, $reward, $transactionId, 0, $availableAt);
				$this->m_core->addNotification($userId, "Your DripOffers Offer #" . $offerId . " is pending approval.", 0);
			}
			echo "ok";
		} else {
			echo "DUP";
		}
	}

	public function offers4all() 
 {
  $secret_key = $this->data['settings']['offers4all_secret_key'];   // Set this one, it is provided to you. 
  $userId = isset($_REQUEST['user_id']) ? $this->db->escape_str($_REQUEST['user_id']) : null; 
  $transactionId = isset($_REQUEST['transaction_id']) ? $this->db->escape_str($_REQUEST['transaction_id']) : null; 
  $reward = isset($_REQUEST['amount']) ? $this->db->escape_str($_REQUEST['amount']) : null * $this->data['settings']['currency_rate']; 
  $userIp = isset($_REQUEST['ip_user']) ? $this->db->escape_str($_REQUEST['ip_user']) : "0.0.0.0"; 
  $signature = isset($_REQUEST['ms']) ? $this->db->escape_str($_REQUEST['ms']) : null;
  $offerId = isset($_REQUEST['offer_id']) ? $this->db->escape_str($_REQUEST['offer_id']) : null; 
  if (md5($transactionId.":".$secret_key) != $signature) { 
   echo "0"; 
   return; 
  } 
 
  $trans = $this->m_offerwall->getTransaction($transactionId, 'Offers4all'); 
  if (!$trans) { 
   $hold = 0; 
   if ($reward > $this->data['settings']['offerwall_min_hold']) { 
    $hold = 3; //set the hold in days in here 
   } 
   if ($hold == 0) { 
    $offerId = $this->m_offerwall->insertTransaction($userId, 'Offers4all', $userIp, $reward, $transactionId, 2, time()); 
    $this->m_offerwall->updateUserBalance($userId, $reward); 
    $this->m_core->addNotification($userId, currencyDisplay($reward, $this->data['settings']) . " from Offers4all Offer #" . $offerId . " was credited to your balance.", 1); 
 
    $user = $this->m_core->getUserFromId($userId); 
    $this->m_core->addExp($user['id'], $this->data['settings']['offerwall_exp_reward']); 
    if (($user['exp'] + $this->data['settings']['offerwall_exp_reward']) >= ($user['level'] + 1) * 100) { 
     $this->m_core->levelUp($user['id']); 
    } 
   } else { 
    $availableAt = time() + $hold * 86400; 
    $offerId = $this->m_offerwall->insertTransaction($userId, 'Offers4all', $userIp, $reward, $transactionId, 0, $availableAt); 
    $this->m_core->addNotification($userId, "Your Offers4all Offer #" . $offerId . " is pending approval.", 0); 
   } 
   echo "200"; 
  } else { 
   echo "0"; 
  } 
 }
 
	public function adwallgate()
	{

		$userId = isset($_REQUEST['subId']) ? $this->db->escape_str($_REQUEST['subId']) : null;
		$transactionId = isset($_REQUEST['transId']) ? $this->db->escape_str($_REQUEST['transId']) : null;
		$reward = isset($_REQUEST['reward']) ? $this->db->escape_str($_REQUEST['reward']) : null;
		$userIp = isset($_REQUEST['userIp']) ? $this->db->escape_str($_REQUEST['userIp']) : "0.0.0.0";
		$signature = isset($_REQUEST['signature']) ? $this->db->escape_str($_REQUEST['signature']) : null;
		if (md5($userId . $transactionId . $reward . $this->data['settings']['adwallgate_secret']) != $signature) {
			echo "ERROR: Signature doesn't match";
			return;
		}
		$reward = $reward * $this->data['settings']['currency_rate'];

		$trans = $this->m_offerwall->getTransaction($transactionId, 'AdWallGate');
		if (!$trans) {
			$hold = 0;
			if ($reward > $this->data['settings']['offerwall_min_hold']) {
				$hold = $this->data['settings']['adwallgate_hold'];
			}
			if ($hold == 0) {
				$offerId = $this->m_offerwall->insertTransaction($userId, 'AdWallGate', $userIp, $reward, $transactionId, 2, time());
				$this->m_offerwall->updateUserBalance($userId, $reward);
				$this->m_core->addNotification($userId, currencyDisplay($reward, $this->data['settings']) . " from AdWallGate Offer #" . $offerId . " was credited to your balance.", 1);

				$user = $this->m_core->getUserFromId($userId);
				$this->m_core->addExp($user['id'], $this->data['settings']['offerwall_exp_reward']);
				if (($user['exp'] + $this->data['settings']['offerwall_exp_reward']) >= ($user['level'] + 1) * 100) {
					$this->m_core->levelUp($user['id']);
				}
			} else {
				$availableAt = time() + $hold * 86400;
				$offerId = $this->m_offerwall->insertTransaction($userId, 'AdWallGate', $userIp, $reward, $transactionId, 0, $availableAt);
				$this->m_core->addNotification($userId, "Your AdWallGate Offer #" . $offerId . " is pending approval.", 0);
			}
			echo "ok";
		} else {
			echo "DUP";
		}
	}
	
public function timewall()
  {
    
    $secret = $this->data['settings']['timewall_app_secret'];

    $userId = isset($_GET['userid']) ? $this->db->escape_str($_GET['userid']) : 2;
    $transactionId = isset($_GET['transactionID']) ? $this->db->escape_str($_GET['transactionID']) : null;
        $revenue = isset($_GET['revenue']) ? $this->db->escape_str($_GET['revenue']) : null;
        $currencyAmount = isset($_GET['currencyAmount']) ? $this->db->escape_str($_GET['currencyAmount']) : null;
    $ipAddress = isset($_GET['ip']) ? $this->db->escape_str($_GET['ip']) : '0.0.0.0';
    $signature = isset($_GET['hash']) ? $this->db->escape_str($_GET['hash']) : null;

    if (hash("sha256", $userId . $revenue . $secret) != $signature) {
      echo 0;
      return;
    }

// First, apply the currency rate
$currencyAmount = $currencyAmount * $this->data['settings']['currency_rate'];

// Then, check if Happy Hour is ON and apply 50% extra
if ($this->data['settings']['happyhour_status'] == 'on') {
    $currencyAmount *= 1.50;  // Add 50% extra to the currency amount
}

    if ($currencyAmount < 0) {
      $this->m_offerwall->reduceUserBalance($userId, abs($currencyAmount));
      $this->m_offerwall->insertTransaction($userId, 'Timewall', $ipAddress, $currencyAmount, $transactionId, 1, time());
      echo 1;
    } else {
      $trans = $this->m_offerwall->getTransaction($transactionId, 'timewall');
      if (!$trans) {
        $hold = 0;
        if ($currencyAmount > $this->data['settings']['offerwall_min_hold']) {
          $hold = $this->data['settings']['timewall_hold'];
        }
        if ($hold == 0) {
          $offerId = $this->m_offerwall->insertTransaction($userId, 'timewall', $ipAddress, $currencyAmount, $transactionId, 2, time());
          $this->m_offerwall->updateUserBalance($userId, $currencyAmount);
          $this->m_core->addNotification($userId, currencyDisplay($currencyAmount, $this->data['settings']) . " from Timewall Offer #" . $offerId . " was credited to your balance.", 1);

          $user = $this->m_core->getUserFromId($userId);
          $this->m_core->addExp($user['id'], $this->data['settings']['offerwall_exp_reward']);
          if (($user['exp'] + $this->data['settings']['offerwall_exp_reward']) >= ($user['level'] + 1) * 100) {
            $this->m_core->levelUp($user['id']);
          }
        } else {
          $availableAt = time() + $hold * 86400;
          $offerId = $this->m_offerwall->insertTransaction($userId, 'Timewall', $ipAddress, $currencyAmount, $transactionId, 0, $availableAt);
          $this->m_core->addNotification($userId, "Your Timewall Offer #" . $offerId . " is pending approval.", 0);
        }
        echo 1;
      } else {
        echo 1;
      }
    }
  }


	public function cryptoad()
	{

		$userId = isset($_REQUEST['subId']) ? $this->db->escape_str($_REQUEST['subId']) : null;
		$transactionId = isset($_REQUEST['transId']) ? $this->db->escape_str($_REQUEST['transId']) : null;
		$reward = isset($_REQUEST['reward']) ? $this->db->escape_str($_REQUEST['reward']) : null;
		$userIp = isset($_REQUEST['userIp']) ? $this->db->escape_str($_REQUEST['userIp']) : "0.0.0.0";
		$signature = isset($_REQUEST['signature']) ? $this->db->escape_str($_REQUEST['signature']) : null;
		if (md5($userId . $transactionId . $reward . $this->data['settings']['cryptoad_secret']) != $signature) {
			echo "ERROR: Signature doesn't match";
			return;
		}
		$reward = $reward * $this->data['settings']['currency_rate'];

		$trans = $this->m_offerwall->getTransaction($transactionId, 'CryptoAd');
		if (!$trans) {
			$hold = 0;
			if ($reward > $this->data['settings']['offerwall_min_hold']) {
				$hold = $this->data['settings']['cryptoad_hold'];
			}
			if ($hold == 0) {
				$offerId = $this->m_offerwall->insertTransaction($userId, 'CryptoAd', $userIp, $reward, $transactionId, 2, time());
				$this->m_offerwall->updateUserBalance($userId, $reward);
				$this->m_core->addNotification($userId, currencyDisplay($reward, $this->data['settings']) . " from CryptoAd Offer #" . $offerId . " was credited to your balance.", 1);

				$user = $this->m_core->getUserFromId($userId);
				$this->m_core->addExp($user['id'], $this->data['settings']['offerwall_exp_reward']);
				if (($user['exp'] + $this->data['settings']['offerwall_exp_reward']) >= ($user['level'] + 1) * 100) {
					$this->m_core->levelUp($user['id']);
				}
			} else {
				$availableAt = time() + $hold * 86400;
				$offerId = $this->m_offerwall->insertTransaction($userId, 'CryptoAd', $userIp, $reward, $transactionId, 0, $availableAt);
				$this->m_core->addNotification($userId, "Your CryptoAd Offer #" . $offerId . " is pending approval.", 0);
			}
			echo "ok";
		} else {
			echo "DUP";
		}
	}
	public function offeroc()
	{

		$userId = isset($_REQUEST['subId']) ? $this->db->escape_str($_REQUEST['subId']) : null;
		$transactionId = isset($_REQUEST['transId']) ? $this->db->escape_str($_REQUEST['transId']) : null;
		$reward = isset($_REQUEST['reward']) ? $this->db->escape_str($_REQUEST['reward']) : null;
		$userIp = isset($_REQUEST['userIp']) ? $this->db->escape_str($_REQUEST['userIp']) : "0.0.0.0";
		$signature = isset($_REQUEST['signature']) ? $this->db->escape_str($_REQUEST['signature']) : null;
		if (md5($userId . $transactionId . $reward . $this->data['settings']['offeroc_secret']) != $signature) {
			echo "ERROR: Signature doesn't match";
			return;
		}
		$reward = $reward * $this->data['settings']['currency_rate'];

		$trans = $this->m_offerwall->getTransaction($transactionId, 'Offeroc');
		if (!$trans) {
			$hold = 0;
			if ($reward > $this->data['settings']['offerwall_min_hold']) {
				$hold = $this->data['settings']['offeroc_hold'];
			}
			if ($hold == 0) {
				$offerId = $this->m_offerwall->insertTransaction($userId, 'Offeroc', $userIp, $reward, $transactionId, 2, time());
				$this->m_offerwall->updateUserBalance($userId, $reward);
				$this->m_core->addNotification($userId, currencyDisplay($reward, $this->data['settings']) . " from Offeroc Offer #" . $offerId . " was credited to your balance.", 1);

				$user = $this->m_core->getUserFromId($userId);
				$this->m_core->addExp($user['id'], $this->data['settings']['offerwall_exp_reward']);
				if (($user['exp'] + $this->data['settings']['offerwall_exp_reward']) >= ($user['level'] + 1) * 100) {
					$this->m_core->levelUp($user['id']);
				}
			} else {
				$availableAt = time() + $hold * 86400;
				$offerId = $this->m_offerwall->insertTransaction($userId, 'Offeroc', $userIp, $reward, $transactionId, 0, $availableAt);
				$this->m_core->addNotification($userId, "Your Offeroc Offer #" . $offerId . " is pending approval.", 0);
			}
			echo "ok";
		} else {
			echo "DUP";
		}
	}
	public function walloffer()
	{

		$userId = isset($_REQUEST['subId']) ? $this->db->escape_str($_REQUEST['subId']) : null;
		$transactionId = isset($_REQUEST['transId']) ? $this->db->escape_str($_REQUEST['transId']) : null;
		$reward = isset($_REQUEST['reward']) ? $this->db->escape_str($_REQUEST['reward']) : null;
		$userIp = isset($_REQUEST['userIp']) ? $this->db->escape_str($_REQUEST['userIp']) : "0.0.0.0";
		$signature = isset($_REQUEST['signature']) ? $this->db->escape_str($_REQUEST['signature']) : null;
		if (md5($userId . $transactionId . $reward . $this->data['settings']['walloffer_secret']) != $signature) {
			echo "ERROR: Signature doesn't match";
			return;
		}
		$reward = $reward * $this->data['settings']['currency_rate'];

		$trans = $this->m_offerwall->getTransaction($transactionId, 'WallOffer');
		if (!$trans) {
			$hold = 0;
			if ($reward > $this->data['settings']['offerwall_min_hold']) {
				$hold = $this->data['settings']['walloffer_hold'];
			}
			if ($hold == 0) {
				$offerId = $this->m_offerwall->insertTransaction($userId, 'WallOffer', $userIp, $reward, $transactionId, 2, time());
				$this->m_offerwall->updateUserBalance($userId, $reward);
				$this->m_core->addNotification($userId, currencyDisplay($reward, $this->data['settings']) . " from WallOffer Offer #" . $offerId . " was credited to your balance.", 1);

				$user = $this->m_core->getUserFromId($userId);
				$this->m_core->addExp($user['id'], $this->data['settings']['offerwall_exp_reward']);
				if (($user['exp'] + $this->data['settings']['offerwall_exp_reward']) >= ($user['level'] + 1) * 100) {
					$this->m_core->levelUp($user['id']);
				}
			} else {
				$availableAt = time() + $hold * 86400;
				$offerId = $this->m_offerwall->insertTransaction($userId, 'WallOffer', $userIp, $reward, $transactionId, 0, $availableAt);
				$this->m_core->addNotification($userId, "Your WallOffer Offer #" . $offerId . " is pending approval.", 0);
			}
			echo "ok";
		} else {
			echo "DUP";
		}
	}
	public function profitswall()
	{

		$userId = isset($_REQUEST['subId']) ? $this->db->escape_str($_REQUEST['subId']) : null;
		$transactionId = isset($_REQUEST['transId']) ? $this->db->escape_str($_REQUEST['transId']) : null;
		$reward = isset($_REQUEST['reward']) ? $this->db->escape_str($_REQUEST['reward']) : null;
		$userIp = isset($_REQUEST['userIp']) ? $this->db->escape_str($_REQUEST['userIp']) : "0.0.0.0";
		$signature = isset($_REQUEST['signature']) ? $this->db->escape_str($_REQUEST['signature']) : null;
		if (md5($userId . $transactionId . $reward . $this->data['settings']['profitswall_secret']) != $signature) {
			echo "ERROR: Signature doesn't match";
			return;
		}
		$reward = $reward * $this->data['settings']['currency_rate'];

		$trans = $this->m_offerwall->getTransaction($transactionId, 'ProfitsWall');
		if (!$trans) {
			$hold = 0;
			if ($reward > $this->data['settings']['offerwall_min_hold']) {
				$hold = $this->data['settings']['profitswall_hold'];
			}
			if ($hold == 0) {
				$offerId = $this->m_offerwall->insertTransaction($userId, 'ProfitsWall', $userIp, $reward, $transactionId, 2, time());
				$this->m_offerwall->updateUserBalance($userId, $reward);
				$this->m_core->addNotification($userId, currencyDisplay($reward, $this->data['settings']) . " from ProfitsWall Offer #" . $offerId . " was credited to your balance.", 1);

				$user = $this->m_core->getUserFromId($userId);
				$this->m_core->addExp($user['id'], $this->data['settings']['offerwall_exp_reward']);
				if (($user['exp'] + $this->data['settings']['offerwall_exp_reward']) >= ($user['level'] + 1) * 100) {
					$this->m_core->levelUp($user['id']);
				}
			} else {
				$availableAt = time() + $hold * 86400;
				$offerId = $this->m_offerwall->insertTransaction($userId, 'ProfitsWall', $userIp, $reward, $transactionId, 0, $availableAt);
				$this->m_core->addNotification($userId, "Your ProfitsWall Offer #" . $offerId . " is pending approval.", 0);
			}
			echo "ok";
		} else {
			echo "DUP";
		}
	}
	public function coinswall()
	{

		$userId = isset($_REQUEST['subId']) ? $this->db->escape_str($_REQUEST['subId']) : null;
		$transactionId = isset($_REQUEST['transId']) ? $this->db->escape_str($_REQUEST['transId']) : null;
		$reward = isset($_REQUEST['reward']) ? $this->db->escape_str($_REQUEST['reward']) : null;
		$userIp = isset($_REQUEST['userIp']) ? $this->db->escape_str($_REQUEST['userIp']) : "0.0.0.0";
		$signature = isset($_REQUEST['signature']) ? $this->db->escape_str($_REQUEST['signature']) : null;
		if (md5($userId . $transactionId . $reward . $this->data['settings']['coinswall_secret']) != $signature) {
			echo "ERROR: Signature doesn't match";
			return;
		}
		$reward = $reward * $this->data['settings']['currency_rate'];

		$trans = $this->m_offerwall->getTransaction($transactionId, 'CoinsWall');
		if (!$trans) {
			$hold = 0;
			if ($reward > $this->data['settings']['offerwall_min_hold']) {
				$hold = $this->data['settings']['coinswall_hold'];
			}
			if ($hold == 0) {
				$offerId = $this->m_offerwall->insertTransaction($userId, 'CoinsWall', $userIp, $reward, $transactionId, 2, time());
				$this->m_offerwall->updateUserBalance($userId, $reward);
				$this->m_core->addNotification($userId, currencyDisplay($reward, $this->data['settings']) . " from CoinsWall Offer #" . $offerId . " was credited to your balance.", 1);

				$user = $this->m_core->getUserFromId($userId);
				$this->m_core->addExp($user['id'], $this->data['settings']['offerwall_exp_reward']);
				if (($user['exp'] + $this->data['settings']['offerwall_exp_reward']) >= ($user['level'] + 1) * 100) {
					$this->m_core->levelUp($user['id']);
				}
			} else {
				$availableAt = time() + $hold * 86400;
				$offerId = $this->m_offerwall->insertTransaction($userId, 'CoinsWall', $userIp, $reward, $transactionId, 0, $availableAt);
				$this->m_core->addNotification($userId, "Your CoinsWall Offer #" . $offerId . " is pending approval.", 0);
			}
			echo "ok";
		} else {
			echo "DUP";
		}
	}
	public function ewall()
	{

		$userId = isset($_REQUEST['subId']) ? $this->db->escape_str($_REQUEST['subId']) : null;
		$transactionId = isset($_REQUEST['transId']) ? $this->db->escape_str($_REQUEST['transId']) : null;
		$reward = isset($_REQUEST['reward']) ? $this->db->escape_str($_REQUEST['reward']) : null;
		$userIp = isset($_REQUEST['userIp']) ? $this->db->escape_str($_REQUEST['userIp']) : "0.0.0.0";
		$signature = isset($_REQUEST['signature']) ? $this->db->escape_str($_REQUEST['signature']) : null;
		if (md5($userId . $transactionId . $reward . $this->data['settings']['ewall_secret']) != $signature) {
			echo "ERROR: Signature doesn't match";
			return;
		}
		$reward = $reward * $this->data['settings']['currency_rate'];

		$trans = $this->m_offerwall->getTransaction($transactionId, 'Ewall');
		if (!$trans) {
			$hold = 0;
			if ($reward > $this->data['settings']['offerwall_min_hold']) {
				$hold = $this->data['settings']['ewall_hold'];
			}
			if ($hold == 0) {
				$offerId = $this->m_offerwall->insertTransaction($userId, 'Ewall', $userIp, $reward, $transactionId, 2, time());
				$this->m_offerwall->updateUserBalance($userId, $reward);
				$this->m_core->addNotification($userId, currencyDisplay($reward, $this->data['settings']) . " from Ewall Offer #" . $offerId . " was credited to your balance.", 1);

				$user = $this->m_core->getUserFromId($userId);
				$this->m_core->addExp($user['id'], $this->data['settings']['offerwall_exp_reward']);
				if (($user['exp'] + $this->data['settings']['offerwall_exp_reward']) >= ($user['level'] + 1) * 100) {
					$this->m_core->levelUp($user['id']);
				}
			} else {
				$availableAt = time() + $hold * 86400;
				$offerId = $this->m_offerwall->insertTransaction($userId, 'Ewall', $userIp, $reward, $transactionId, 0, $availableAt);
				$this->m_core->addNotification($userId, "Your Ewall Offer #" . $offerId . " is pending approval.", 0);
			}
			echo "ok";
		} else {
			echo "DUP";
		}
	}
	public function bitcotasks()
	{

		$userId = isset($_REQUEST['subId']) ? $this->db->escape_str($_REQUEST['subId']) : null;
		$transactionId = isset($_REQUEST['transId']) ? $this->db->escape_str($_REQUEST['transId']) : null;
		$reward = isset($_REQUEST['reward']) ? $this->db->escape_str($_REQUEST['reward']) : null;
		$payout = isset($_REQUEST['payout']) ? $this->db->escape_str($_REQUEST['payout']) : null;
		$userIp = isset($_REQUEST['userIp']) ? $this->db->escape_str($_REQUEST['userIp']) : "0.0.0.0";
		$signature = isset($_REQUEST['signature']) ? $this->db->escape_str($_REQUEST['signature']) : null;
		if (md5($userId . $transactionId . $reward . $this->data['settings']['bitcotasks_secret']) != $signature) {
			echo "ERROR: Signature doesn't match";
			return;
		}
		$reward = $reward * $this->data['settings']['currency_rate'];
// Then, check if Happy Hour is ON and apply 50% extra
if ($this->data['settings']['happyhour_status'] == 'on') {
    $reward *= 1.50;  // Add 50% extra to the currency amount
}

		$trans = $this->m_offerwall->getTransaction($transactionId, 'BitcoTasks');
		if (!$trans) {
			$hold = 0;
			if ($reward > $this->data['settings']['offerwall_min_hold']) {
				$hold = $this->data['settings']['bitcotasks_hold'];
			}
			if ($hold == 0) {
				$offerId = $this->m_offerwall->insertTransaction($userId, 'BitcoTasks', $userIp, $reward, $transactionId, 2, time(), '', $payout);
				$this->m_offerwall->updateUserBalance($userId, $reward);
				$this->m_core->addNotification($userId, currencyDisplay($reward, $this->data['settings']) . " from BitcoTasks Offer #" . $offerId . " was credited to your balance.", 1);

				$user = $this->m_core->getUserFromId($userId);
				$this->m_core->addExp($user['id'], $this->data['settings']['offerwall_exp_reward']);
				if (($user['exp'] + $this->data['settings']['offerwall_exp_reward']) >= ($user['level'] + 1) * 100) {
					$this->m_core->levelUp($user['id']);
				}
			} else {
				$availableAt = time() + $hold * 86400;
				$offerId = $this->m_offerwall->insertTransaction($userId, 'BitcoTasks', $userIp, $reward, $transactionId, 0, $availableAt, '', $payout);
				$this->m_core->addNotification($userId, "Your BitcoTasks Offer #" . $offerId . " is pending approval.", 0);
			}
			echo "ok";
		} else {
			echo "DUP";
		}
	}
	public function bitlabs()
    {

        $newtwork = 'bitlabs';
        $userId = isset($_GET['user']) ? $this->db->escape_str($_GET['user']) : null;
        $transactionId = isset($_GET['tx']) ? $this->db->escape_str($_GET['tx']) : null;
        $reward = isset($_GET['reward']) ? $this->db->escape_str($_GET['reward']) : null;
        $action = isset($_GET['status']) ? $this->db->escape_str($_GET['status']) : null;
        $action = ($action == 'RECONCILIATION') ? 2 : 1;
        $userIp = "0.0.0.0";
        $offer_name = isset($_GET['of_name']) ? $this->db->escape_str($_GET['of_name']) : null;
        $payout = isset($_GET['payout']) ? $this->db->escape_str($_GET['payout']) : 0;

        $reward = $reward * $this->data['settings']['currency_rate'];

        $trans = $this->m_offerwall->getTransaction($transactionId, $newtwork);
        if ($action == 2) {
            $this->m_offerwall->reduceUserBalance($userId, abs($reward));
            $this->m_offerwall->insertTransaction($userId, $newtwork, $userIp, $reward, $transactionId, 1, time(), $offer_name, $payout);
            echo "1";
        } else {
            if (!$trans) {
                $hold = 0;
                if ($reward > $this->data['settings']['offerwall_min_hold']) {
                    $hold = $this->data['settings']['bitlabs_hold'];
                }
                if ($hold == 0) {
                    $offerId = $this->m_offerwall->insertTransaction($userId, $newtwork, $userIp, $reward, $transactionId, 2, time(), $offer_name, $payout);
                    $this->m_offerwall->updateUserBalance($userId, $reward);
                    $this->m_core->addNotification($userId, currencyDisplay($reward, $this->data['settings']) . " from $newtwork Offer #" . $offerId . " was credited to your balance.", 1);

                    $user = $this->m_core->getUserFromId($userId);
                    $this->m_core->addExp($user['id'], $this->data['settings']['offerwall_exp_reward']);
                    if (($user['exp'] + $this->data['settings']['offerwall_exp_reward']) >= ($user['level'] + 1) * 100) {
                        $this->m_core->levelUp($user['id']);
                    }
                } else {
                    $availableAt = time() + $hold * 86400;
                    $offerId = $this->m_offerwall->insertTransaction($userId, $newtwork, $userIp, $reward, $transactionId, 0, $availableAt, $offer_name, $payout);
                    $this->m_core->addNotification($userId, "Your $newtwork Offer #" . $offerId . " is pending approval.", 0);
                }
                echo "1";
            } else {
                echo "0";
            }
        }
    }
	
	public function bitswall()
	{
		// if (!in_array($this->input->ip_address(), $this->data['whitelist_ips']['bitswall'])) {
		// 	echo 'ok';
		// 	die();
		// }
		$userId = isset($_REQUEST['subId']) ? $this->db->escape_str($_REQUEST['subId']) : null;
		$transactionId = isset($_REQUEST['transId']) ? $this->db->escape_str($_REQUEST['transId']) : null;
		$reward = isset($_REQUEST['reward']) ? $this->db->escape_str($_REQUEST['reward']) : null;
		$userIp = isset($_REQUEST['userIp']) ? $this->db->escape_str($_REQUEST['userIp']) : "0.0.0.0";
		$signature = isset($_REQUEST['signature']) ? $this->db->escape_str($_REQUEST['signature']) : null;
		if (md5($userId . $transactionId . $reward . $this->data['settings']['bitswall_secret']) != $signature) {
			echo "ERROR: Signature doesn't match";
			return;
		}
		$reward = $reward * $this->data['settings']['currency_rate'];

		$trans = $this->m_offerwall->getTransaction($transactionId, 'Bitswall');
		if (!$trans) {
			$hold = 0;
			if ($reward > $this->data['settings']['offerwall_min_hold']) {
				$hold = $this->data['settings']['bitswall_hold'];
			}
			if ($hold == 0) {
				$offerId = $this->m_offerwall->insertTransaction($userId, 'Bitswall', $userIp, $reward, $transactionId, 2, time());
				$this->m_offerwall->updateUserBalance($userId, $reward);
				$this->m_core->addNotification($userId, currencyDisplay($reward, $this->data['settings']) . " from Bitswall Offer #" . $offerId . " was credited to your balance.", 1);

				$user = $this->m_core->getUserFromId($userId);
				$this->m_core->addExp($user['id'], $this->data['settings']['offerwall_exp_reward']);
				if (($user['exp'] + $this->data['settings']['offerwall_exp_reward']) >= ($user['level'] + 1) * 100) {
					$this->m_core->levelUp($user['id']);
				}
			} else {
				$availableAt = time() + $hold * 86400;
				$offerId = $this->m_offerwall->insertTransaction($userId, 'Bitswall', $userIp, $reward, $transactionId, 0, $availableAt);
				$this->m_core->addNotification($userId, "Your Bitswall Offer #" . $offerId . " is pending approval.", 0);
			}
			echo "ok";
		} else {
			echo "DUP";
		}
	}

	public function monlix()
	{
		// UPDATE YOUR SECRET KEY
		$secretKey = $this->data['settings']['monlix_secret'];
		$hold = $this->data['settings']['monlix_hold'];
		$userId = isset($_GET['userId']) ? $this->db->escape_str($_GET['userId']) : null;
		$userIp = isset($_GET['userIp']) ? $this->db->escape_str($_GET['userIp']) : "0.0.0.0";
		$transactionId = isset($_GET['transactionId']) ? $this->db->escape_str($_GET['transactionId']) : null;
		$reward = isset($_GET['rewardValue']) ? $this->db->escape_str($_GET['rewardValue']) : null;
		$payout = isset($_GET['payout']) ? $this->db->escape_str($_GET['payout']) : null;
		$offer_name = isset($_GET['taskName']) ? $this->db->escape_str($_GET['taskName']) : null;
		$action = isset($_GET['status']) ? $this->db->escape_str($_GET['status']) : null;
		$signature = isset($_GET['secretKey']) ? $this->db->escape_str($_GET['secretKey']) : null;
		if ($secretKey != $signature) {
			echo "ERROR: Signature doesn't match";
			return;
		}
		$reward = $reward * $this->data['settings']['currency_rate'];
// Then, check if Happy Hour is ON and apply 50% extra
if ($this->data['settings']['happyhour_status'] == 'on') {
    $reward *= 1.50;  // Add 50% extra to the currency amount
}

		$trans = $this->m_offerwall->getTransaction($transactionId, 'Monlix');
		if ($action == 2) {
			$this->m_offerwall->reduceUserBalance($userId, abs($reward));
			$this->m_offerwall->insertTransaction($userId, 'Monlix', $userIp, $reward, $transactionId, 1, time(), $offer_name, $payout);
			echo "OK";
		} else {
			if (!$trans) {
				$hold = 0;
				if ($reward > $this->data['settings']['offerwall_min_hold']) {
					$hold = $this->data['settings']['monlix_hold'];
				}
				if ($hold == 0) {
					$offerId = $this->m_offerwall->insertTransaction($userId, 'Monlix', $userIp, $reward, $transactionId, 2, time(), $offer_name, $payout);
					$this->m_offerwall->updateUserBalance($userId, $reward);
					$this->m_core->addNotification($userId, currencyDisplay($reward, $this->data['settings']) . " from Monlix Offer #" . $offerId . " was credited to your balance.", 1);
					$user = $this->m_core->getUserFromId($userId);
					$this->m_core->addExp($user['id'], $this->data['settings']['offerwall_exp_reward']);
					if (($user['exp'] + $this->data['settings']['offerwall_exp_reward']) >= ($user['level'] + 1) * 100) {
						$this->m_core->levelUp($user['id']);
					}
				} else {
					$availableAt = time() + $hold * 86400;
					$offerId = $this->m_offerwall->insertTransaction($userId, 'Monlix', $userIp, $reward, $transactionId, 0, $availableAt, $offer_name, $payout);
					$this->m_core->addNotification($userId, "Your Monlix Offer #" . $offerId . " is pending approval.", 0);
				}
				echo "OK";
			} else {
				echo "DUP";
			}
		}
	}
	public function adgatemedia()
    {

        $newtwork = 'Adgatemedia';
        if (!in_array($this->input->ip_address(), $this->data['whitelist_ips']['adgatemedia'])) {
            die();
        }
        
        $userId = isset($_GET['user_id']) ? $this->db->escape_str($_GET['user_id']) : null;
        $transactionId = isset($_GET['transaction_id']) ? $this->db->escape_str($_GET['transaction_id']) : null;
        $reward = isset($_GET['reward']) ? $this->db->escape_str($_GET['reward']) : null;
        $action = isset($_GET['status']) ? ($_GET['status'] == 'rejected' ? 2 : ($_GET['status'] == 'approved' ? 1 : null)) : null;
        $userIp = isset($_GET['ip']) ? $this->db->escape_str($_GET['ip']) : "0.0.0.0";
        $offer_name = isset($_GET['of_name']) ? $this->db->escape_str($_GET['of_name']) : null;
        $payout = isset($_GET['payout'])? $this->db->escape_str($_GET['payout']) : null;

        $reward = $reward * $this->data['settings']['currency_rate'];

        $trans = $this->m_offerwall->getTransaction($transactionId, $newtwork);
        if ($action == 2) {
            $this->m_offerwall->reduceUserBalance($userId, abs($reward));
            $this->m_offerwall->insertTransaction($userId, $newtwork, $userIp, $reward, $transactionId, 1, time(), $offer_name, $payout);
            echo "1";
        } else {
            if (!$trans) {
                $hold = 0;
                if ($reward > $this->data['settings']['offerwall_min_hold']) {
                    $hold = $this->data['settings']['adgate_hold'];
                }
                if ($hold == 0 && $action == 1) {
                    $offerId = $this->m_offerwall->insertTransaction($userId, $newtwork, $userIp, $reward, $transactionId, 2, time(), $offer_name, $payout);
                    $this->m_offerwall->updateUserBalance($userId, $reward);
                    $this->m_core->addNotification($userId, currencyDisplay($reward, $this->data['settings']) . " from $newtwork Offer #" . $offerId . " was credited to your balance.", 1);

                    $user = $this->m_core->getUserFromId($userId);
                    $this->m_core->addExp($user['id'], $this->data['settings']['offerwall_exp_reward']);
                    if (($user['exp'] + $this->data['settings']['offerwall_exp_reward']) >= ($user['level'] + 1) * 100) {
                        $this->m_core->levelUp($user['id']);
                    }
                } else {
                    $availableAt = time() + $hold * 86400;
                    $offerId = $this->m_offerwall->insertTransaction($userId, $newtwork, $userIp, $reward, $transactionId, $offer_name, 0, $availableAt, $offer_name, $payout);
                    $this->m_core->addNotification($userId, "Your $newtwork Offer #" . $offerId . " is pending approval.", 0);
                }
                echo "1";
            } else {
                echo "0";
            }
        }
    }
	public function earnetize()
    {
        $userId = isset($_GET['userid']) ? $this->db->escape_str($_GET['userid']) : null;
        $transactionId = 'earnetize';
        $reward = isset($_GET['user_amount']) ? $this->db->escape_str($_GET['user_amount']) : null;
        $password = isset($_GET['password']) ? $this->db->escape_str($_GET['password']) : null;
        $userIp = isset($_GET['ip_address']) ? $this->db->escape_str($_GET['ip_address']) : "0.0.0.0";
        $offer_name = isset($_GET['offer_name']) ? $this->db->escape_str($_GET['offer_name']) : null;
        $payout = isset($_GET['payout']) ? $this->db->escape_str($_GET['payout']) : null;
        if ($this->data['settings']['earnetize_password'] !== $password) {
            echo "ERROR: Password doesn't match";
            return;
        }

        $reward = $reward * $this->data['settings']['currency_rate'];

        if ($reward < 0) {
            $this->m_offerwall->reduceUserBalance($userId, abs($reward));
            $this->m_offerwall->insertTransaction($userId, 'earnetize', $userIp, $reward, $transactionId, 1, time(), $offer_name, $payout);
            echo "1";
        } else {
            $hold = 0;
            if ($reward > $this->data['settings']['offerwall_min_hold']) {
                $hold = $this->data['settings']['earnetize_hold'];
            }
            if ($hold == 0) {
                $offerId = $this->m_offerwall->insertTransaction($userId, 'earnetize', $userIp, $reward, $transactionId, 2, time(), $offer_name, $payout);
                $this->m_offerwall->updateUserBalance($userId, $reward);
                $this->m_core->addNotification($userId, currencyDisplay($reward, $this->data['settings']) . " from earnetize Offer #" . $offerId . " was credited to your balance.", 1);

                $user = $this->m_core->getUserFromId($userId);
                $this->m_core->addExp($user['id'], $this->data['settings']['offerwall_exp_reward']);
                if (($user['exp'] + $this->data['settings']['offerwall_exp_reward']) >= ($user['level'] + 1) * 100) {
                    $this->m_core->levelUp($user['id']);
                }
            } else {
                $availableAt = time() + $hold * 86400;
                $offerId = $this->m_offerwall->insertTransaction($userId, 'earnetize', $userIp, $reward, $transactionId, 0, $availableAt, $offer_name, $payout);
                $this->m_core->addNotification($userId, "Your Earnetize Offer #" . $offerId . " is pending approval.", 0);
            }
            echo "1";
        }
    }
    
    
    public function taskwall()
    {

        $userId = isset($_GET['userid']) ? $this->db->escape_str($_GET['userid']) : null;
        $transactionId = 'taskwall';
        $reward = isset($_GET['user_amount']) ? $this->db->escape_str($_GET['user_amount']) : null;
        $password = isset($_GET['password']) ? $this->db->escape_str($_GET['password']) : null;
        $userIp = isset($_GET['ip_address']) ? $this->db->escape_str($_GET['ip_address']) : "0.0.0.0";
        $offer_name = isset($_GET['offer_name']) ? $this->db->escape_str($_GET['offer_name']) : null;
        $payout = isset($_GET['payout']) ? $this->db->escape_str($_GET['payout']) : null;
        if ($this->data['settings']['taskwall_password'] !== $password) {
            echo "ERROR: Password doesn't match";
            return;
        }

        $reward = $reward * $this->data['settings']['currency_rate'];

// Then, check if Happy Hour is ON and apply 50% extra
if ($this->data['settings']['happyhour_status'] == 'on') {
    $reward *= 1.50;  // Add 50% extra to the currency amount
}


        if ($reward < 0) {
            $this->m_offerwall->reduceUserBalance($userId, abs($reward));
            $this->m_offerwall->insertTransaction($userId, 'taskwall', $userIp, $reward, $transactionId, 1, time(), $offer_name, $payout);
            echo "1";
        } else {
            $hold = 0;
            if ($reward > $this->data['settings']['offerwall_min_hold']) {
                $hold = $this->data['settings']['taskwall_hold'];
            }
            if ($hold == 0) {
                $offerId = $this->m_offerwall->insertTransaction($userId, 'taskwall', $userIp, $reward, $transactionId, 2, time(), $offer_name, $payout);
                $this->m_offerwall->updateUserBalance($userId, $reward);
                $this->m_core->addNotification($userId, currencyDisplay($reward, $this->data['settings']) . " from taskwall Offer #" . $offerId . " was credited to your balance.", 1);

                $user = $this->m_core->getUserFromId($userId);
                $this->m_core->addExp($user['id'], $this->data['settings']['offerwall_exp_reward']);
                if (($user['exp'] + $this->data['settings']['offerwall_exp_reward']) >= ($user['level'] + 1) * 100) {
                    $this->m_core->levelUp($user['id']);
                }
            } else {
                $availableAt = time() + $hold * 86400;
                $offerId = $this->m_offerwall->insertTransaction($userId, 'taskwall', $userIp, $reward, $transactionId, 0, $availableAt, $offer_name, $payout);
                $this->m_core->addNotification($userId, "Your taskwall Offer #" . $offerId . " is pending approval.", 0);
            }
            echo "1";
        }
    }
    
    public function upwall()
    {
        $newtwork = "Upwall";
        $userId = isset($_GET['userid']) ? $this->db->escape_str($_GET['userid']) : null;
        $transactionId = isset($_GET['transaction']) ? $this->db->escape_str($_GET['transaction']) : null;
        $reward = isset($_GET['user_amount']) ? $this->db->escape_str($_GET['user_amount']) : null;
        $password = isset($_GET['password']) ? $this->db->escape_str($_GET['password']) : null;
        $userIp = isset($_GET['ip_address']) ? $this->db->escape_str($_GET['ip_address']) : "0.0.0.0";
        $offer_name = isset($_GET['offer_name']) ? $this->db->escape_str($_GET['offer_name']) : null;
        $payout = isset($_GET['payout']) ? $this->db->escape_str($_GET['payout']) : null;
        if ($this->data['settings']['upwall_password'] !== $password) {
            echo "ERROR: Password doesn't match";
            return;
        }

        $reward = $reward * $this->data['settings']['currency_rate'];
        $trans = $this->m_offerwall->getTransaction($transactionId, $newtwork);
        if ($reward < 0) {
            $this->m_offerwall->reduceUserBalance($userId, abs($reward));
            $this->m_offerwall->insertTransaction($userId, $newtwork, $userIp, $reward, $transactionId, 1, time(), $offer_name, $payout);
            echo "1";
        } else {
            if (!$trans) {
                $hold = 0;
                if ($reward > $this->data['settings']['offerwall_min_hold']) {
                    $hold = $this->data['settings']['upwall_hold'];
                }
                if ($hold == 0 && $reward > 0) {
                    $offerId = $this->m_offerwall->insertTransaction($userId, $newtwork, $userIp, $reward, $transactionId, 2, time(), $offer_name, $payout);
                    $this->m_offerwall->updateUserBalance($userId, $reward);
                    $this->m_core->addNotification($userId, currencyDisplay($reward, $this->data['settings']) . " from $newtwork Offer #" . $offerId . " was credited to your balance.", 1);

                    $user = $this->m_core->getUserFromId($userId);
                    $this->m_core->addExp($user['id'], $this->data['settings']['offerwall_exp_reward']);
                    if (($user['exp'] + $this->data['settings']['offerwall_exp_reward']) >= ($user['level'] + 1) * 100) {
                        $this->m_core->levelUp($user['id']);
                    }
                } else {
                    $availableAt = time() + $hold * 86400;
                    $offerId = $this->m_offerwall->insertTransaction($userId, $newtwork, $userIp, $reward, $transactionId, $offer_name, 0, $availableAt, $offer_name, $payout);
                    $this->m_core->addNotification($userId, "Your $newtwork Offer #" . $offerId . " is pending approval.", 0);
                }
                echo "1";
            } else {
                echo "0";
            }
        }
    }

    public function edotiva()
    {

        $newtwork = 'Edotiva';
        
        $userId = isset($_GET['subId']) ? $this->db->escape_str($_GET['subId']) : null;
        $transactionId = isset($_GET['transId']) ? $this->db->escape_str($_GET['transId']) : null;
        $reward = isset($_GET['reward']) ? $this->db->escape_str($_GET['reward']) : null;
        $action = isset($_GET['status']) ? $this->db->escape_str($_GET['status']) : null;
        $userIp = isset($_GET['userIp']) ? $this->db->escape_str($_GET['userIp']) : "0.0.0.0";
        $offer_name = isset($_GET['offer_name']) ? $this->db->escape_str($_GET['offer_name']) : null;
        $payout = isset($_GET['payout'])? $this->db->escape_str($_GET['payout']) : null;
        $signature = isset($_GET['signature']) ? $this->db->escape_str($_GET['signature']) : null;

        if(md5($userId . $transactionId . $reward . $this->data['settings']['edotiva_secret_key']) !== $signature) {
             echo "ERROR: Signature doesn't match";
            return;
        }


        $reward = $reward * $this->data['settings']['currency_rate'];

        $trans = $this->m_offerwall->getTransaction($transactionId, $newtwork);
       if ($action == 2) {
            $this->m_offerwall->reduceUserBalance($userId, abs($reward));
            $this->m_offerwall->insertTransaction($userId, $newtwork, $userIp, $reward, $transactionId, 1, time(), $offer_name, $payout);
            echo "1";
        } else {
            if (!$trans) {
                $hold = 0;
                if ($reward > $this->data['settings']['offerwall_min_hold']) {
                    $hold = $this->data['settings']['edotiva_hold'];
                }
                if ($hold == 0 && $action == 1) {
                    $offerId = $this->m_offerwall->insertTransaction($userId, $newtwork, $userIp, $reward, $transactionId, 2, time(), $offer_name, $payout);
                    $this->m_offerwall->updateUserBalance($userId, $reward);
                    $this->m_core->addNotification($userId, currencyDisplay($reward, $this->data['settings']) . " from $newtwork Offer #" . $offerId . " was credited to your balance.", 1);

                    $user = $this->m_core->getUserFromId($userId);
                    $this->m_core->addExp($user['id'], $this->data['settings']['offerwall_exp_reward']);
                    if (($user['exp'] + $this->data['settings']['offerwall_exp_reward']) >= ($user['level'] + 1) * 100) {
                        $this->m_core->levelUp($user['id']);
                    }
                } else {
                    $availableAt = time() + $hold * 86400;
                    $offerId = $this->m_offerwall->insertTransaction($userId, $newtwork, $userIp, $reward, $transactionId, $offer_name, 0, $availableAt, $offer_name, $payout);
                    $this->m_core->addNotification($userId, "Your $newtwork Offer #" . $offerId . " is pending approval.", 0);
                }
                echo "1";
            } else {
                echo "0";
            }
        }
    }
    public function adswedmedia()
    {

        $newtwork = 'Adswedmedia';
        
        $userId = isset($_GET['subId']) ? $this->db->escape_str($_GET['subId']) : null;
        $transactionId = isset($_GET['transId']) ? $this->db->escape_str($_GET['transId']) : null;
        $reward = isset($_GET['reward']) ? $this->db->escape_str($_GET['reward']) : null;
        $action = isset($_GET['status']) ? $this->db->escape_str($_GET['status']) : null;
        $userIp = isset($_GET['userIp']) ? $this->db->escape_str($_GET['userIp']) : "0.0.0.0";
        $offer_name = isset($_GET['offer_name']) ? $this->db->escape_str($_GET['offer_name']) : null;
        $payout = isset($_GET['payout'])? $this->db->escape_str($_GET['payout']) : null;
        $signature = isset($_GET['signature']) ? $this->db->escape_str($_GET['signature']) : null;

        if(md5($userId . $transactionId . $reward . $this->data['settings']['edotiva_secret_key']) !== $signature) {
             echo "ERROR: Signature doesn't match";
            return;
        }


        $reward = $reward * $this->data['settings']['currency_rate'];

        $trans = $this->m_offerwall->getTransaction($transactionId, $newtwork);
       if ($action == 2) {
            $this->m_offerwall->reduceUserBalance($userId, abs($reward));
            $this->m_offerwall->insertTransaction($userId, $newtwork, $userIp, $reward, $transactionId, 1, time(), $offer_name, $payout);
            echo "1";
        } else {
            if (!$trans) {
                $hold = 0;
                if ($reward > $this->data['settings']['offerwall_min_hold']) {
                    $hold = $this->data['settings']['adswedmedia_hold'];
                }
                if ($hold == 0 && $action == 1) {
                    $offerId = $this->m_offerwall->insertTransaction($userId, $newtwork, $userIp, $reward, $transactionId, 2, time(), $offer_name, $payout);
                    $this->m_offerwall->updateUserBalance($userId, $reward);
                    $this->m_core->addNotification($userId, currencyDisplay($reward, $this->data['settings']) . " from $newtwork Offer #" . $offerId . " was credited to your balance.", 1);

                    $user = $this->m_core->getUserFromId($userId);
                    $this->m_core->addExp($user['id'], $this->data['settings']['offerwall_exp_reward']);
                    if (($user['exp'] + $this->data['settings']['offerwall_exp_reward']) >= ($user['level'] + 1) * 100) {
                        $this->m_core->levelUp($user['id']);
                    }
                } else {
                    $availableAt = time() + $hold * 86400;
                    $offerId = $this->m_offerwall->insertTransaction($userId, $newtwork, $userIp, $reward, $transactionId, $offer_name, 0, $availableAt, $offer_name, $payout);
                    $this->m_core->addNotification($userId, "Your $newtwork Offer #" . $offerId . " is pending approval.", 0);
                }
                echo "1";
            } else {
                echo "0";
            }
        }
    }
    
    public function adrevmedia()
    {

        $newtwork = 'Adrevmedia';
        
        $userId = isset($_GET['subId']) ? $this->db->escape_str($_GET['subId']) : null;
        $transactionId = isset($_GET['transId']) ? $this->db->escape_str($_GET['transId']) : null;
        $reward = isset($_GET['reward']) ? $this->db->escape_str($_GET['reward']) : null;
        $action = isset($_GET['status']) ? $this->db->escape_str($_GET['status']) : null;
        $userIp = isset($_GET['userIp']) ? $this->db->escape_str($_GET['userIp']) : "0.0.0.0";
        $offer_name = isset($_GET['offer_name']) ? $this->db->escape_str($_GET['offer_name']) : null;
        $payout = isset($_GET['payout'])? $this->db->escape_str($_GET['payout']) : null;
        $signature = isset($_GET['signature']) ? $this->db->escape_str($_GET['signature']) : null;

        if(md5($userId . $transactionId . $reward . $this->data['settings']['adrevmedia_secret_key']) !== $signature) {
             echo "ERROR: Signature doesn't match";
            return;
        }


        $reward = $reward * $this->data['settings']['currency_rate'];

        $trans = $this->m_offerwall->getTransaction($transactionId, $newtwork);
       if ($action == 2) {
            $this->m_offerwall->reduceUserBalance($userId, abs($reward));
            $this->m_offerwall->insertTransaction($userId, $newtwork, $userIp, $reward, $transactionId, 1, time(), $offer_name, $payout);
            echo "1";
        } else {
            if (!$trans) {
                $hold = 0;
                if ($reward > $this->data['settings']['offerwall_min_hold']) {
                    $hold = $this->data['settings']['adrevmedia_hold'];
                }
                if ($hold == 0 && $action == 1) {
                    $offerId = $this->m_offerwall->insertTransaction($userId, $newtwork, $userIp, $reward, $transactionId, 2, time(), $offer_name, $payout);
                    $this->m_offerwall->updateUserBalance($userId, $reward);
                    $this->m_core->addNotification($userId, currencyDisplay($reward, $this->data['settings']) . " from $newtwork Offer #" . $offerId . " was credited to your balance.", 1);

                    $user = $this->m_core->getUserFromId($userId);
                    $this->m_core->addExp($user['id'], $this->data['settings']['offerwall_exp_reward']);
                    if (($user['exp'] + $this->data['settings']['offerwall_exp_reward']) >= ($user['level'] + 1) * 100) {
                        $this->m_core->levelUp($user['id']);
                    }
                } else {
                    $availableAt = time() + $hold * 86400;
                    $offerId = $this->m_offerwall->insertTransaction($userId, $newtwork, $userIp, $reward, $transactionId, $offer_name, 0, $availableAt, $offer_name, $payout);
                    $this->m_core->addNotification($userId, "Your $newtwork Offer #" . $offerId . " is pending approval.", 0);
                }
                echo "1";
            } else {
                echo "0";
            }
        }
    }
    public function adfleetmedia()
    {

        $newtwork = 'Adfleetmedia';
        
        $userId = isset($_GET['subId']) ? $this->db->escape_str($_GET['subId']) : null;
        $transactionId = isset($_GET['transId']) ? $this->db->escape_str($_GET['transId']) : null;
        $reward = isset($_GET['reward']) ? $this->db->escape_str($_GET['reward']) : null;
        $action = isset($_GET['status']) ? $this->db->escape_str($_GET['status']) : null;
        $userIp = isset($_GET['userIp']) ? $this->db->escape_str($_GET['userIp']) : "0.0.0.0";
        $offer_name = isset($_GET['offer_name']) ? $this->db->escape_str($_GET['offer_name']) : null;
        $payout = isset($_GET['payout'])? $this->db->escape_str($_GET['payout']) : null;
        $signature = isset($_GET['signature']) ? $this->db->escape_str($_GET['signature']) : null;

        if(md5($userId . $transactionId . $reward . $this->data['settings']['adfleetmedia_secret_key']) !== $signature) {
             echo "ERROR: Signature doesn't match";
            return;
        }


        $reward = $reward * $this->data['settings']['currency_rate'];

        $trans = $this->m_offerwall->getTransaction($transactionId, $newtwork);
       if ($action == 2) {
            $this->m_offerwall->reduceUserBalance($userId, abs($reward));
            $this->m_offerwall->insertTransaction($userId, $newtwork, $userIp, $reward, $transactionId, 1, time(), $offer_name, $payout);
            echo "1";
        } else {
            if (!$trans) {
                $hold = 0;
                if ($reward > $this->data['settings']['offerwall_min_hold']) {
                    $hold = $this->data['settings']['adfleetmedia_hold'];
                }
                if ($hold == 0 && $action == 1) {
                    $offerId = $this->m_offerwall->insertTransaction($userId, $newtwork, $userIp, $reward, $transactionId, 2, time(), $offer_name, $payout);
                    $this->m_offerwall->updateUserBalance($userId, $reward);
                    $this->m_core->addNotification($userId, currencyDisplay($reward, $this->data['settings']) . " from $newtwork Offer #" . $offerId . " was credited to your balance.", 1);

                    $user = $this->m_core->getUserFromId($userId);
                    $this->m_core->addExp($user['id'], $this->data['settings']['offerwall_exp_reward']);
                    if (($user['exp'] + $this->data['settings']['offerwall_exp_reward']) >= ($user['level'] + 1) * 100) {
                        $this->m_core->levelUp($user['id']);
                    }
                } else {
                    $availableAt = time() + $hold * 86400;
                    $offerId = $this->m_offerwall->insertTransaction($userId, $newtwork, $userIp, $reward, $transactionId, $offer_name, 0, $availableAt, $offer_name, $payout);
                    $this->m_core->addNotification($userId, "Your $newtwork Offer #" . $offerId . " is pending approval.", 0);
                }
                echo "1";
            } else {
                echo "0";
            }
        }
    }
	public function adgem()
	{
	// Define your AdGem credentials
define('ADGEM_WHITELIST_IP', '43.225.204.70');
define('ADGEM_POSTBACK_KEY', ''. $this->data['settings']['adgem_api'] .'');

// Verify the request IP matches the whitelisted IP
if ($_SERVER['REMOTE_ADDR'] !== ADGEM_WHITELIST_IP) {
    http_response_code(403);
    exit('Error: ' . $_SERVER['REMOTE_ADDR'] . ' does not match the whitelisted IP address.');
}

// Get the full request URL
$protocol = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? "https" : "http";
$request_url = "$protocol://$_SERVER[HTTP_HOST]$_SERVER[REQUEST_URI]";

// Parse the URL and query string
$parsed_url = parse_url($request_url);
parse_str($parsed_url['query'], $query_string);

// Get the verifier value
$verifier = $query_string['verifier'] ?? null;
if (is_null($verifier)) {
    http_response_code(422);
    exit("Error: missing verifier");
}

// Rebuild URL without the verifier
unset($query_string['verifier']);
$hashless_url = $protocol.'://'.$parsed_url['host'].$parsed_url['path'].'?'.http_build_query($query_string, "", "&", PHP_QUERY_RFC3986);

// Calculate the hash and verify it matches the provided one
$calculated_hash = hash_hmac('sha256', $hashless_url, ADGEM_POSTBACK_KEY);
if ($calculated_hash !== $verifier) {
    http_response_code(422);
    exit('Error: invalid verifier');
}

// Extract necessary parameters from the request
$app_id = $_GET['appid'] ?? null;
$user_id = $_GET['userid'] ?? null;
$payout = $_GET['payout'] ?? null;

// Perform further processing here, such as updating user balance, logging transactions, etc.

if ($app_id && $user_id) {
	            $hold = 0;
				if ($reward > $this->data['settings']['offerwall_min_hold']) {
					$hold = $this->data['settings']['adgem_hold'];
				}
				if ($hold == 0) {
					$offerId = $this->m_offerwall->insertTransaction($userId, 'Adgem', $userIp, $reward, $transactionId, 2, time(), '', $payout);
					$this->m_offerwall->updateUserBalance($userId, $reward);
					$this->m_core->addNotification($userId, currencyDisplay($reward, $this->data['settings']) . " from Adgem Offer #" . $offerId . " was credited to your balance.", 1);
					$user = $this->m_core->getUserFromId($userId);
					$this->m_core->addExp($user['id'], $this->data['settings']['offerwall_exp_reward']);
					if (($user['exp'] + $this->data['settings']['offerwall_exp_reward']) >= ($user['level'] + 1) * 100) {
						$this->m_core->levelUp($user['id']);
					}
				} else {
					$availableAt = time() + $hold * 86400;
					$offerId = $this->m_offerwall->insertTransaction($userId, 'Adgem', $userIp, $reward, $transactionId, 0, $availableAt, '', $payout);
					$this->m_core->addNotification($userId, "Your Adgem Offer #" . $offerId . " is pending approval.", 0);
				}
}

// Respond with OK status
http_response_code(200);
echo 'OK';
	}

	public function notik()
	{
		$userId = isset($_GET['user_id']) ? $this->db->escape_str($_GET['user_id']) : null;
		$reward = isset($_GET['amount']) ? $this->db->escape_str($_GET['amount']) : null;
		$payout = isset($_GET['payout']) ? $this->db->escape_str($_GET['payout']) : null;
		$offer_name = isset($_GET['offer_name']) ? $this->db->escape_str($_GET['offer_name']) : null;
		$transactionId = isset($_GET['txn_id']) ? $this->db->escape_str($_GET['txn_id']) : null;
		$hash = $_GET['hash'];

		$protocol = (isset($_SERVER["HTTPS"]) && $_SERVER["HTTPS"] === "on") ? "https" : "http";
		$url = "$protocol://$_SERVER[HTTP_HOST]$_SERVER[REQUEST_URI]";
		$urlWithoutHash = substr($url, 0, -strlen("&hash=$hash"));
		$generatedHash = hash_hmac("sha1", $urlWithoutHash, $this->data['settings']['notik_secret']);
		if ($generatedHash != $hash) {
			echo 0;
			die();
		}
		$reward *= $this->data['settings']['currency_rate'];
		$trans = $this->m_offerwall->getTransaction($transactionId, 'Notik');
		if (!$trans) {
			$hold = 0;
			if ($reward > $this->data['settings']['offerwall_min_hold']) {
				$hold = $this->data['settings']['notik_hold'];
			}
			if ($hold == 0) {
				$offerId = $this->m_offerwall->insertTransaction($userId, 'Notik', '', $reward, $transactionId, 2, time(), $offer_name, $payout);
				$this->m_offerwall->updateUserBalance($userId, $reward);
				$this->m_core->addNotification($userId, currencyDisplay($reward, $this->data['settings']) . " from Notik Offer #" . $offerId . " was credited to your balance.", 1);
				$user = $this->m_core->getUserFromId($userId);
				$this->m_core->addExp($user['id'], $this->data['settings']['offerwall_exp_reward']);
				if (($user['exp'] + $this->data['settings']['offerwall_exp_reward']) >= ($user['level'] + 1) * 100) {
					$this->m_core->levelUp($user['id']);
				}
			} else {
				$availableAt = time() + $hold * 86400;
				$offerId = $this->m_offerwall->insertTransaction($userId, 'Notik', '', $reward, $payout, $transactionId, 0, $availableAt, $offer_name, $payout);
				$this->m_core->addNotification($userId, "Your Notik Offer #" . $offerId . " is pending approval.", 0);
			}
			echo 1;
		} else {
			echo 0;
		}
	}

	public function ccpayment()
    {
        
        http_response_code(200);
       
        $postJson = json_decode(file_get_contents('php://input'), true);
		

        $headers = getallheaders();
        $appId = $headers["Appid"];
        $reqtimestamp = $headers["Timestamp"];
        $sign = $headers["Sign"];
        $signature = hash("sha256", $appId . $this->data['settings']['ccpayment_secret'] . $reqtimestamp . json_encode($postJson));
        
        if ($sign != $signature OR $appId != $this->data['settings']['ccpayment_appid'] OR $postJson['pay_status'] != "success")
        {
            echo 'invalid';
            die();
        }
        $trans = isset($postJson['order_id']) ? $this->db->escape_str($postJson['order_id']) : null;
        $moid = isset($postJson['extend']['merchant_order_id']) ? $this->db->escape_str($postJson['extend']['merchant_order_id']) : null;
        $user = explode("_", $moid)[0];
		$reward = $this->db->escape_str($postJson['paid_amount']) * $this->db->escape_str($postJson['token_rate']);
        // $amount = $reward * $this->data['settings']['faucetpay_deposite_rate'];
        
		$this->m_deposit->addDeposit($user, $reward, $trans, 4, 'Confirmed');
		$this->m_deposit->updateUser($user, $reward);
        echo "Success";
        
    }
    
    
    
    
    
             public function Revtoo()
  {
      $secret = "278ea16813f962f3a598ac79aab777fe
"; // UPDATE YOUR SECRET KEY
      $hold = 1;  // UPDATE HOLD DAYS IF YOU USE HOLD
      $minHold= 0.1; // Reward Lower than this amount will not be hold
     
      $userId = isset($_GET['subId']) ? $this->db->escape_str($_GET['subId']) : null;
      $transactionId = isset($_GET['transId']) ? $this->db->escape_str($_GET['transId']) : null;
      $reward = isset($_GET['reward']) ? $this->db->escape_str($_GET['reward']) : null;
      $userIp = isset($_GET['userIp']) ? $this->db->escape_str($_GET['userIp']) : "0.0.0.0";
      $signature = isset($_GET['signature']) ? $this->db->escape_str($_GET['signature']) : null;
      $status = isset($_GET['status']) ? $this->db->escape_str($_GET['status']) : null;
      if (md5($userId . $transactionId . $reward . $secret) != $signature) {
          echo "ERROR: Signature doesn't match";
          return;
      }
     
    
      $trans = $this->m_offerwall->getTransaction($transactionId, 'Revtoo');
      if (!$trans) {
         $hold = 0;
         if ($reward > $minHold) {
             $hold = 1; // UPDATE HOLD DAYS Which you Use for hold
         }
        
            if ($status == 2) {
                $this->m_offerwall->reduceUserBalance($userId, abs($reward));
                $this->m_offerwall->insertTransaction($userId, 'Revtoo', $userIp, $reward, $transactionId, 1, time());
                echo "OK";
            } else {
                  if ($hold == 0) {
                      $offerId = $this->m_offerwall->insertTransaction($userId, 'Revtoo', $userIp, $reward, $transactionId, 2, time());
                      $this->m_offerwall->updateUserBalance($userId, $reward);
                      $this->m_core->addNotification($userId, currency($reward, $this->data['settings']['currency_rate']) . " from Excentiv Offer #" . $offerId . " was credited to your balance.", 1);
                     
                      $user = $this->m_core->get_user_from_id($userId);
                      $this->m_core->addExp($user['id'], $this->data['settings']['offerwall_exp_reward']);
                      if (($user['exp'] + $this->data['settings']['offerwall_exp_reward']) >= ($user['level'] + 1) * 100) {
                          $this->m_core->levelUp($user['id']);
                      }
                  } else {
                      $availableAt = time() + $hold * 86400;
                      $offerId = $this->m_offerwall->insertTransaction($userId, 'Revtoo', $userIp, $reward, $transactionId, 0, $availableAt);
                      $this->m_core->addNotification($userId, "Your Revtoo Offer #" . $offerId . " is pending approval.", 0);
                  }
                  echo "OK";
              } 
        } else {
              echo "DUP"; 
        }
     }

    
    

	public function pollfish()
	{
		$transactionId = isset($_GET['tx_id']) ? $this->db->escape_str($_GET['tx_id']) : null;
		$reward = isset($_GET['reward_value']) ? $this->db->escape_str($_GET['reward_value']) : null;
		$status = isset($_GET['status']) ? $this->db->escape_str($_GET['status']) : null;
		$userId = isset($_GET['request_uuid']) ? $this->db->escape_str($_GET['request_uuid']) : null;
		$userIp = isset($_GET['user_ip']) ? $this->db->escape_str($_GET['user_ip']) : "not available";
		$signature = isset($_GET['signature']) ? $this->db->escape_str($_GET['signature']) : "null";
		$debug = isset($_GET['debug']) ? $this->db->escape_str($_GET['debug']) : "null";

		$cpa = rawurldecode($_GET["cpa"]);
		$device_id = rawurldecode($_GET["device_id"]);
		$reward_name = rawurldecode($_GET["reward_name"]);
		$timestamp = rawurldecode($_GET["timestamp"]);

		$data = $cpa . ":" . $device_id;
		if (!empty($userId)) {
			$data = $data . ":" . $userId;
		}
		$data = $data . ":" . $reward_name . ":" . $reward . ":" . $status . ":" . $timestamp . ":" . $transactionId;
		$reward *= $this->data['settings']['currency_rate'];

		$computedSignature = base64_encode(hash_hmac("sha1", $data, $this->data['settings']['pollfish_secret'], true));
		if ($signature == $computedSignature) {
			if ($status == 'eligible') {
				$hold = 0;
				if ($reward > $this->data['settings']['offerwall_min_hold']) {
					$hold = $this->data['settings']['pollfish_hold'];
				}
				if ($hold == 0) {
					$offerId = $this->m_offerwall->insertTransaction($userId, 'Pollfish', $userIp, $reward, $transactionId, 2, time());
					$this->m_offerwall->updateUserBalance($userId, $reward);
					$this->m_core->addNotification($userId, currencyDisplay($reward, $this->data['settings']) . " from Pollfish Offer #" . $offerId . " was credited to your balance.", 1);

					$user = $this->m_core->getUserFromId($userId);
					$this->m_core->addExp($user['id'], $this->data['settings']['offerwall_exp_reward']);
					if (($user['exp'] + $this->data['settings']['offerwall_exp_reward']) >= ($user['level'] + 1) * 100) {
						$this->m_core->levelUp($user['id']);
					}
				} else {
					$availableAt = time() + $this->data['settings']['pollfish_hold'] * 86400;
					$offerId = $this->m_offerwall->insertTransaction($userId, 'Pollfish', $userIp, $reward, $transactionId, 0, $availableAt);
					$this->m_core->addNotification($userId, "Your Pollfish Offer #" . $offerId . " is pending approval.", 0);
				}
				echo "1";
			} else {
				$this->m_offerwall->insertTransaction($userId, 'Pollfish', $userIp, $reward, $transactionId, 1, time());
			}
		}
	}
	
	public function hugoffers()
	{
		// Load HugOffers models
		$this->load->helper('hugoffers_helper');
		$this->load->model('M_hugoffers');
		
		// Debug mode - if ?debug=1 is passed, show detailed info
		if (isset($_GET['debug']) && $_GET['debug'] == '1') {
			$this->debug_hugoffers();
			return;
		}
		
		// Get postback parameters - check both GET and POST
		$clickId = isset($_GET['aff_sub']) ? $this->db->escape_str($_GET['aff_sub']) : (isset($_POST['aff_sub']) ? $this->db->escape_str($_POST['aff_sub']) : null);
		$offerId = isset($_GET['offer_id']) ? $this->db->escape_str($_GET['offer_id']) : (isset($_POST['offer_id']) ? $this->db->escape_str($_POST['offer_id']) : null);
		$payout = isset($_GET['payout']) ? floatval($_GET['payout']) : (isset($_POST['payout']) ? floatval($_POST['payout']) : 0);
		$eventName = isset($_GET['event_name']) ? $this->db->escape_str($_GET['event_name']) : (isset($_POST['event_name']) ? $this->db->escape_str($_POST['event_name']) : 'install');
		$eventValue = isset($_GET['event_value']) ? $this->db->escape_str($_GET['event_value']) : (isset($_POST['event_value']) ? $this->db->escape_str($_POST['event_value']) : null);
		$userIp = isset($_GET['ip']) ? $this->db->escape_str($_GET['ip']) : (isset($_POST['ip']) ? $this->db->escape_str($_POST['ip']) : $this->input->ip_address());
		
		// Log all incoming parameters for debugging
		log_message('info', 'HugOffers Webhook Received: ' . json_encode($_REQUEST));
		log_message('info', 'HugOffers Webhook GET: ' . json_encode($_GET));
		log_message('info', 'HugOffers Webhook POST: ' . json_encode($_POST));
		
		// Log the extracted parameters
		log_message('info', 'HugOffers Extracted Parameters - Click ID: ' . $clickId . ', Offer ID: ' . $offerId . ', Payout: ' . $payout);
		
		// Validate required parameters
		if (!$clickId || $payout <= 0) {
			log_message('error', 'HugOffers Webhook: Missing required parameters - clickId: ' . $clickId . ', payout: ' . $payout);
			echo "ERROR: Missing required parameters";
			return;
		}
		
		// Extract user ID from click ID format: userId_offerId_timestamp_random
		$userId = $this->extractUserIdFromClickId($clickId);
		
		// Log the extraction process for debugging
		log_message('info', 'HugOffers Webhook: Click ID: ' . $clickId . ', Extracted User ID: ' . $userId);
		
		if (!$userId) {
			log_message('error', 'HugOffers Webhook: Could not determine user ID from click ID: ' . $clickId);
			echo "ERROR: Could not determine user ID";
			return;
		}
		
		// Get USD conversion rate
		$usd_rate = isset($this->data['settings']['hugoffers_usd_rate']) ? floatval($this->data['settings']['hugoffers_usd_rate']) : 1000;
		
		// Convert payout to site currency
		$reward = $payout * $usd_rate;
		
		// Apply Happy Hour bonus if enabled
		if ($this->data['settings']['happyhour_status'] == 'on') {
			$reward *= 1.50; // Add 50% extra during happy hour
		}
		
		// Create transaction ID from click ID for duplicate checking
		$transactionId = $clickId;
		
		// Check for duplicate transaction
		$trans = $this->m_offerwall->getTransaction($transactionId, 'HugOffers');
		if (!$trans) {
			$hold = 0;
			if ($reward > $this->data['settings']['offerwall_min_hold']) {
				$hold = $this->data['settings']['hugoffers_hold'];
			}
			
			if ($hold == 0) {
				// Instant credit
				$offerwallId = $this->m_offerwall->insertTransaction($userId, 'HugOffers', $userIp, $reward, $transactionId, 2, time(), $offerId, $payout);
				$this->m_offerwall->updateUserBalance($userId, $reward);
				$this->m_core->addNotification($userId, currencyDisplay($reward, $this->data['settings']) . " from HugOffers Offer #" . $offerwallId . " was credited to your balance.", 1);
				
				// Add experience points
				$user = $this->m_core->getUserFromId($userId);
				$this->m_core->addExp($user['id'], $this->data['settings']['offerwall_exp_reward']);
				if (($user['exp'] + $this->data['settings']['offerwall_exp_reward']) >= ($user['level'] + 1) * 100) {
					$this->m_core->levelUp($user['id']);
				}
				
				// Update offer conversion count
				if ($offerId) {
					$this->M_hugoffers->incrementConversion($offerId);
				}
				
				// Log conversion
				log_hugoffers_conversion($clickId, $offerId, $payout, $userId);
				
			} else {
				// Pending approval
				$availableAt = time() + $hold * 86400;
				$offerwallId = $this->m_offerwall->insertTransaction($userId, 'HugOffers', $userIp, $reward, $transactionId, 0, $availableAt, $offerId, $payout);
				$this->m_core->addNotification($userId, "Your HugOffers Offer #" . $offerwallId . " is pending approval.", 0);
			}
			
			echo "1"; // Success response
		} else {
			echo "0"; // Duplicate transaction
		}
	}
	
	/**
	 * Extract user ID from click ID
	 * Click ID format: userId_offerId_timestamp_random
	 */
	private function extractUserIdFromClickId($clickId) {
		// Log the click ID for debugging
		log_message('info', 'Extracting user ID from click ID: ' . $clickId);
		
		// Extract user ID from click ID format: userId_offerId_timestamp_random
		$parts = explode('_', $clickId);
		
		log_message('info', 'Click ID parts: ' . json_encode($parts));
		
		if (count($parts) >= 4 && is_numeric($parts[0])) {
			$userId = intval($parts[0]);
			log_message('info', 'Successfully extracted user ID: ' . $userId);
			return $userId;
		}
		
		// Fallback: Extract from URL parameters if available
		if (isset($_REQUEST['user_id'])) {
			$userId = intval($_REQUEST['user_id']);
			log_message('info', 'Using user_id from REQUEST: ' . $userId);
			return $userId;
		}
		
		// Fallback: Use a custom tracking parameter
		if (isset($_REQUEST['custom1'])) {
			$userId = intval($_REQUEST['custom1']);
			log_message('info', 'Using custom1 from REQUEST: ' . $userId);
			return $userId;
		}
		
		log_message('error', 'Could not extract user ID from click ID: ' . $clickId);
		return null;
	}
	
	/**
	 * Debug method for HugOffers webhook
	 */
	private function debug_hugoffers() {
		echo "<h2>HugOffers Webhook Debug</h2>";
		echo "<p><strong>Request Method:</strong> " . $_SERVER['REQUEST_METHOD'] . "</p>";
		echo "<p><strong>Request URI:</strong> " . $_SERVER['REQUEST_URI'] . "</p>";
		
		echo "<h3>All Request Parameters:</h3>";
		echo "<pre>" . json_encode($_REQUEST, JSON_PRETTY_PRINT) . "</pre>";
		
		echo "<h3>All GET Parameters:</h3>";
		echo "<pre>" . json_encode($_GET, JSON_PRETTY_PRINT) . "</pre>";
		
		echo "<h3>All POST Parameters:</h3>";
		echo "<pre>" . json_encode($_POST, JSON_PRETTY_PRINT) . "</pre>";
		
		// Test click ID extraction
		if (isset($_REQUEST['aff_sub'])) {
			$clickId = $_REQUEST['aff_sub'];
			$userId = $this->extractUserIdFromClickId($clickId);
			
			echo "<h3>Click ID Extraction Test:</h3>";
			echo "<p><strong>Click ID:</strong> {$clickId}</p>";
			echo "<p><strong>Extracted User ID:</strong> " . ($userId !== null ? $userId : 'null') . "</p>";
			
			if ($userId) {
				echo "<p style='color: green;'>✅ User ID extraction successful!</p>";
			} else {
				echo "<p style='color: red;'>❌ User ID extraction failed!</p>";
			}
		}
	}
	
	
	
	public function offery() {
        
        $newtwork = 'Offery';
        $secret = $this->data['settings']['offery_secret'];
        
        // echo '<pre>';
        // print_r($_REQUEST);
        // echo '</pre>';
        // exit;
        
        $userId = isset($_REQUEST['subId']) ? $this->db->escape_str($_REQUEST['subId']) : null;
        $transactionId =  isset($_REQUEST['transId']) ? $this->db->escape_str($_REQUEST['transId']) : null;
        $reward = isset($_REQUEST['reward']) ? $this->db->escape_str($_REQUEST['reward']) : null;
        $action = isset($_REQUEST['status']) ? $this->db->escape_str($_REQUEST['status']) : null;
        $userIp = isset($_REQUEST['userIp']) ? $this->db->escape_str($_REQUEST['userIp']) : "0.0.0.0";
        $offer_name = isset($_REQUEST['offer_name']) ? $this->db->escape_str($_REQUEST['offer_name']) : null;
        $payout = isset($_REQUEST['payout'])? $this->db->escape_str($_REQUEST['payout']) : null;
        $signature = isset($_REQUEST['signature']) ? $this->db->escape_str($_REQUEST['signature']) : null;
        
        
        if(md5($userId.$transactionId.$reward.$secret) != $signature) {
            echo "ERROR: Signature doesn't match";
            return;
        }
        
        $reward = $reward * $this->data['settings']['currency_rate'];

        $trans = $this->m_offerwall->getTransaction($transactionId, $newtwork);
        
       if ($action == 2) {
            $this->m_offerwall->reduceUserBalance($userId, abs($reward));
            $this->m_offerwall->insertTransaction($userId, $newtwork, $userIp, $reward, $transactionId, 1, time(), $offer_name, $payout);
            echo "1";
        } else {
            if (!$trans) {
                $hold = 0;
                if ($reward > $this->data['settings']['offerwall_min_hold']) {
                    $hold = $this->data['settings']['adrevmedia_hold'];
                }
                if ($hold == 0 && $action == 1) {
                    $offerId = $this->m_offerwall->insertTransaction($userId, $newtwork, $userIp, $reward, $transactionId, 2, time(), $offer_name, $payout);
                    $this->m_offerwall->updateUserBalance($userId, $reward);
                    $this->m_core->addNotification($userId, currencyDisplay($reward, $this->data['settings']) . " from $newtwork Offer #" . $offerId . " was credited to your balance.", 1);

                    $user = $this->m_core->getUserFromId($userId);
                    $this->m_core->addExp($user['id'], $this->data['settings']['offerwall_exp_reward']);
                    if (($user['exp'] + $this->data['settings']['offerwall_exp_reward']) >= ($user['level'] + 1) * 100) {
                        $this->m_core->levelUp($user['id']);
                    }
                } else {
                    $availableAt = time() + $hold * 86400;
                    $offerId = $this->m_offerwall->insertTransaction($userId, $newtwork, $userIp, $reward, $transactionId, $offer_name, 0, $availableAt, $offer_name, $payout);
                    $this->m_core->addNotification($userId, "Your $newtwork Offer #" . $offerId . " is pending approval.", 0);
                }
                echo "1";
            } else {
                echo "0";
            }
        }
    }
	
	
	
	
}
